% MOOSE EXAMPLE
% 
% Example 2 revisited: Ellipsoidal approximation
% Christian A. Larsson and Mariette Annergren, "Applications Oriented Input
% Design - Optimal input design for control", to be published in 
% Control Systems Magazine
%
% The example aims to illustrate a minimal input power design for a second
% order FIR system that is to be controlled by a P-controller. Because the
% example is simple, the numerical solution can be compared with the
% analytical solution.
%
% True system
% y(t) = theta1*u(t-1) + theta(2)u(t-2) + e(t)
%
% P Controller
%             beta^2
% K = -----------------------,  0 < beta <= 1
%     beta*theta(1)- theta(2)
%
% beta is a design parameter of the controller.
%
% COPYRIGHT 2012.
% Christian Larsson, Mariette Annergren.
% LAST EDITED: 2012-06-19

clc
clear variables
close all

% Setup the true system and P-controller parameters
theta0  = [1 0.8]';
le      = 1;
G0      = tf([0 theta0'],1,1,'variable','z^-1');
H0      = tf(1,1,1,'variable','z^-1');

% Controller tuning, also affects input design
beta    = 0.3;

% Hessian is calculated numerically from Vapp.m
VappHessNum = hessian(@(theta)Vapp(theta,[theta0;beta]),theta0);

% Optimal input design
gamma = 100;
mooseBegin
    model G0 H0 le
    application constraints
        ellipsoid(VappHessNum,gamma)
    identification constraints
        spectrum    phiU = AR(2)
        probability alpha = 0.95
        numSamples  N = 100
    objective   minimize(inputPower)
mooseEnd

% Nominal closed loop system for comparison
a1 = theta0(1)*beta^2/(beta*theta0(1)-theta0(2));
a2 = theta0(2)*beta^2/(beta*theta0(1)-theta0(2));
Gc = tf(1,[1 a1 a2],1,'variable','z^-1');

% Plot results
figure
subplot(2,2,[1 2])
    bodemag(Gc,mooseProblem.spectralFactor); xlim([1e-2 pi]);
    title('Bode plot','FontSize',18,'LineWidth',2)
    legend('Nominal closed loop','\Phi_u','Location','NorthWest')
subplot(2,2,3)
    p1 = pole(Gc);
    plot(real(p1),imag(p1),'bx','MarkerSize',10,'LineWidth',2)
    axis([-1.2 1.2 -1.2 1.2]); axis square; grid on;
    title('Closed loop poles','FontSize',18,'LineWidth',2)
subplot(2,2,4)
    p2 = pole(mooseProblem.spectralFactor);
    plot(real(p2),imag(p2),'bx','MarkerSize',10,'LineWidth',2)
    axis([-1.2 1.2 -1.2 1.2]); axis square; grid on
    title('\Phi_u poles','FontSize',18,'LineWidth',2)