function Vapp   = Vapp_MPC(theta_est)
% AUTHOR:       Mariette Annergren
% FUNCTION:     Vapp_MPC
% DESCRIPTION:  Given the estimated system and the true system, Vapp is
%               constructed for N data samples. An MPC is used to limit the 
%               affect of a unit step disturbance. The corresponding output
%               signals for the true and estimated systems are recorded and
%               used in Vapp.
% INPUT:        estimated parameters
% OUTPUT:       Vapp
%% Parameters
theta_est = theta_est';
N     = 10;
theta = [10 -9];
nb    = length(theta);
%% True system
% x(k+1) = A*x(k)+B*u(k)
% y(k)   = C*x(k)+d
A        = diag(ones(nb-1,1),1);
B        = zeros(nb,1);
B(nb,1)  = 1;
C        = theta(end:-1:1);
ref      = 0;
umax     = 1;
%% Estimated system
C_est    = theta_est(end:-1:1);
%% Prediction horizon
T         = 10;
%% "Unknown" disturbances
dbar      = 1; 
sigma     = 0;
%% Initializing true system
x0        = zeros(nb,1);
xhist     = zeros(nb,N);
uhist     = zeros(N,1);
yhist     = zeros(N,1);
errorhist = zeros(N,1);
error     = (dbar+sigma*randn)*ones(T-1,1);
%% Initializing estimated system
x_est0        = zeros(nb,1);
x_esthist     = zeros(nb,N);
u_esthist     = zeros(N,1);
y_esthist     = zeros(N,1);
error_esthist = zeros(N,1);
error_est     = (dbar+sigma*randn)*ones(T-1,1);
%% Optimization algorithm
x     = x0;
x_est = x_est0;
for i = 1:N   
    cvx_quiet(true)
    cvx_begin
        variables X(nb,T) U(1,T-1)
        X(:,2:T)   == A*X(:,1:T-1)+B*U;
        X(:,1)     == x;
%         abs(U)     <= umax;      
        minimize (norm(C*X(:,1:T-1)+error'-ref))
    cvx_end
    cvx_clear
    cvx_quiet(true)
    cvx_begin
        variables X_est(nb,T) U_est(1,T-1)
        X_est(:,2:T)   == A*X_est(:,1:T-1)+B*U_est;
        X_est(:,1)     == x_est;
%         abs(U_est)     <= umax;
        minimize (norm(C_est*X_est(:,1:T-1)+error_est'-ref))
    cvx_end
    cvx_clear

    % Updating true system
    u          = U(1);
    FIR_output = C*x(1:nb);
    z          = FIR_output+dbar;
    y          = z+sigma*randn;
    error      = (y-FIR_output)*ones(T-1,1);
    xhist(:,i) = x;
    yhist(i)   = y;
    uhist(i)   = u;
    errorhist(i) = error(1);
    x          = A*x+B*u;
    % Updating estimated system
    u_est           = U_est(1);
    FIR_output_true = C*x_est(1:nb);
    z_true          = FIR_output_true+dbar;
    y_true          = z_true+sigma*randn;
    FIR_output_est  = C_est*x_est(1:nb);
    error_est       = (y_true-FIR_output_est)*ones(T-1,1);
    x_esthist(:,i)  = x_est;
    y_esthist(i)    = y_true;
    u_esthist(i)    = u_est;
    error_esthist(i)= error_est(1);
    x_est           = A*x_est+B*u_est;
end
%% Calculating Vapp
Vapp           = 1/N*sum((y_esthist-yhist).^2);
