% Estimating FIR System Controlled by MPC
% Example from:
% Bo Wahlberg, Hkan Hjalmarsson and Mariette Annergren, "On Optimal Input 
% Design in System Identification for Control",
% Proceedings IEEE Conference on Decision and Control, pp. 5548-5552, 2010.
clc
clear all
close all
%% Optimal input design
% True system
theta0          = [10 -9]';
le              = 1;
G0              = tf([0 theta0'],1,1,'variable','z^-1');
H0              = tf(1,1,1,'variable','z^-1');
% You can choose to either define VappBiss and use that as the first
% argument into ellipsoid() or you can send in the m-function of Vapp. Here
% the m-function is
% Vapp_MPC
% and VappBiss is
% [0.032958530889123   0.030176143998619
%  0.030176143998619   0.031306825672633].
VappBiss = [0.032958530889123   0.030176143998619
 0.030176143998619   0.031306825672633];
mooseBegin
    model       G0 H0 le   
    application constraints
        ellipsoid(VappBiss , 100)
    identification constraints
        spectrum    phiU  = FIR(40)
        probability alpha = 0.95
        numSamples  N     = 100 
    objective   minimize(inputPower) 
mooseEnd
%% Identification experiments (check of theory)
VsiBiss   = 2*mooseProblem.informationMatrix;
chi2      = chi2inv(alpha,length(theta0));
uFilter   = mooseProblem.spectralFactor;
uOpt      = lsim(uFilter,randn(1,N));
A         = [1]; 
B         = [0 theta0'];
numPoints = 0;
numExp    = 100;
thetaHat  = zeros(numExp,2);
for i = 1:numExp
    e             = sqrt(le)*randn(N,1);
    y             = filter(B,A,uOpt)+e;
    z             = iddata(y,uOpt);
    m             = pem(z,[0 2 0 0 0 1]);
    thetaHat(i,:) = m.b(2:end);
    if 0.5*(thetaHat(i,:)-theta0')*VsiBiss*(thetaHat(i,:)'-theta0)...
       <= le*chi2/N;
        numPoints = numPoints+1;
    end
end
fracFulfilled = numPoints/numExp
fracDesired   = alpha
%% Plots of resulting regions and estimated points
figure(1)
hold on
ellipse(VsiBiss*N/(2*le*chi2),theta0,'k');
ellipse(0.5*mooseProblem.accuracy*mooseProblem.constraints{1}.H,theta0,'b');
plot(thetaHat(:,1),thetaHat(:,2),'ro')
title('Blue: application region. Black: identification region. Red: estimates')
    