classdef oidModelSs < oidModel
    % State space models used in the input design toolbox.
    % Model is constructed from LTI state space for plant and noise
    % processe. By default, the matrix coefficients are treated as
    % independent parameters. Conflicts of nominal values of user specified
    % parameters results in last given nominal value being assigned.
    %
    % M: x(t+1) = F(theta)x(t) + G(theta)u(t) + Ke(t)
    %      y(t) = H(theta)x(t) + e(t)
    % F,G,H are matrices.
    % e is a white Gaussian process with covariance Re.
    properties
        F = oidParameter();
        G = oidParameter();
        H = oidParameter();
        K = oidParameter();
%         Re;
    end
    
    properties (Dependent = true, SetAccess = private)
        n;
        theta0;
    end
    
    properties (Access = private)
        param;
        dA;
        dB;
        dC;
        dK;
    end
    
    methods
        function obj = oidModelSs(F,G,H,K,Re,Ts,varargin)
            %             if G.Ts == 0 || H.Ts == 0;
            %                 throw(MException('','MOOSE only supports discrete time models'));
            %             end
            obj = obj@oidModel(Re,Ts);
            [p m] = size(G);
            %!!Dimension check incomplete!!
            if p ~= size(F,1) || size(F,1) ~= size(F,2) || p ~= size(H,2) || ...
                    p ~= size(K,1)
                throw(MException('','Dimensions are wrong'));
            end
            % Initialize matrices
            obj.F = repmat(oidParameter(0,0),size(F));
            obj.G = repmat(oidParameter(0,0),size(G));
            obj.H = repmat(oidParameter(0,0),size(H));
            obj.K = repmat(oidParameter(0,0),size(K));
            % Default parameter indices
            if numel(varargin) == 0 ||...
               numel(varargin) == 8 % <-Quick fix for custom derivatives
                % Parameter count
                nT = nnz(F) + nnz(G) + nnz(H) + nnz(K);
                obj.param = oidParameter(zeros(nT,1),1:nT); 
                % Construct matrices
                i = find(F);
                for j = 1:nnz(F)
                    obj.param(j).nominalValue = F(i(j));
                    obj.F(i(j)) = obj.param(j);
                end
                i = find(G);
                for k = 1:nnz(G)
                    obj.param(k+j).nominalValue = G(i(k));
                    obj.G(i(k)) = obj.param(k+j);
                end
                i = find(H);
                for l = 1:nnz(H)
                    obj.param(l+k+j).nominalValue = H(i(l));
                    obj.H(i(l)) = obj.param(l+k+j);
                end
                i = find(K);
                for n = 1:nnz(K)
                    obj.param(n+l+k+j).nominalValue = K(i(n));
                    obj.K(i(n)) = obj.param(n+l+k+j);
                end
                clear i;
            elseif numel(varargin) == 4
                % Parameter count
                nT = max([max(max(varargin{1})),max(max(varargin{2})),...
                          max(max(varargin{3})),max(max(varargin{4}))]);
                      
                obj.param = oidParameter(zeros(nT,1),1:nT);
                % Construct matrices
                i = find(F);
                for j = 1:nnz(F)
                    if varargin{1}(i(j)) == 0
                        obj.F(i(j)) = oidParameter(F(i(j)),0);
                    else
                        obj.param(varargin{1}(i(j))).nominalValue = F(i(j));
                        obj.F(i(j)) = obj.param(varargin{1}(i(j)));
                    end
                end
                i = find(G);
                for j = 1:nnz(G)
                    if varargin{2}(i(j)) == 0
                        obj.G(i(j)) = oidParameter(G(i(j)),0);
                    else
                        obj.param(varargin{2}(i(j))).nominalValue = G(i(j));
                        obj.G(i(j)) = obj.param(varargin{2}(i(j)));
                    end
                end
                i = find(H);
                for j = 1:nnz(H)
                    if varargin{3}(i(j)) == 0
                        obj.H(i(j)) = oidParameter(H(i(j)),0);
                    else
                        obj.param(varargin{3}(i(j))).nominalValue = H(i(j));
                        obj.H(i(j)) = obj.param(varargin{3}(i(j)));
                    end
                end
                i = find(K);
                for j = 1:nnz(K)
                    if varargin{4}(i(j)) == 0
                        obj.H(i(j)) = oidParameter(H(i(j)),0);
                    else
                        obj.param(varargin{4}(i(j))).nominalValue = K(i(j));
                        obj.K(i(j)) = obj.param(varargin{4}(i(j)));
                    end
                end
                clear i;
            else
                throw(MException(...
                    '','MOOSE::oidModelSs::Wrong number of input arguments'));
            end
            % BEGIN QUICK FIX CUSTOM DERIVATIVES %
            if numel(varargin) == 8
                % Set derivatives
                obj.dA = varargin{5};
                obj.dB = varargin{6};
                obj.dC = varargin{7};
                obj.dK = varargin{8};
                
            end
            % END QUICK FIX CUSTOM DERIVATIVES %
        end
        
        function n = get.n(obj)
            if isempty(obj.dA)
                n = numel(obj.param);
            % BEGIN QUICK FIX CUSTOM DERIVATIVES %
            else
                n = size(obj.dA,3);
            % END QUICK FIX CUSTOM DERIVATIVES %
            end
        end
        function theta0 = get.theta0(obj)
            theta0 = zeros(numel(obj.param),1);
            for j = 1:numel(obj.param)
                theta0(j) = obj.param(j).nominalValue;
            end
        end
        
        M = ss(obj);
        [G H] = tf(obj);
        dG = dG(obj,n);
        dH = dH(obj,n);
        bode(obj);
        bodemag(obj);
    end
    
    methods (Access = private)
        [I n] = paramIndices(~,coeff,n,isMonic);
    end
end

% Copyright 2011 Mariette Annergren and Christian A. Larsson
% See the file COPYING.txt for full copyright information.