classdef oidModelTf < oidModel
    % Transfer function models used in the input design toolbox.
    % Model is constructed from LTI transfer functions for plant and noise
    % processe. By default, the transfer function coefficients are treated as
    % independent parameters. Conflicts of nominal values of user specified
    % parameters results in last given nominal value being assigned.
    %
    % M: y(t) = G(q^-1,theta)u(t) + H(q^-1,theta)
    % G and H are oidTf objects
    properties
        G = oidTf;
        H = oidTf;
    end
    
    properties (Dependent = true, SetAccess = private)
        n;
        theta0;
    end
    
    properties (Access = private)
        param;
    end
    
    methods
        function obj = oidModelTf(G,H,le,varargin)
            if ~isa(G,'tf'); G = tf(G,'variable','z^-1'); end;
            if ~isa(H,'tf'); H = tf(H,'variable','z^-1'); end;
            if G.Ts == 0 || H.Ts == 0;
                throw(MException('','MOOSE only supports discrete time models'));
            end
            if G.den{1}(1)~=1 || H.num{1}(1)~=1 || H.den{1}(1)~=1
                throw(MException('','C, D and F polynomials must be monic'));
            end
            obj = obj@oidModel(le,G.Ts);
            [p m] = size(G);
            if [p p] ~= size(H)
                throw(MException('','Dimensions of H are wrong'));
            end
            
            % Default parameter indices
            if numel(varargin) == 0
                % Parameter count
                nT = 0;
                for j = 1:p
                    for k = 1:m
                        nT = nT + nnz(G.num{j,k})+max([nnz(G.den{j,k})-1 0]);
                    end
                    for k = 1:p
                        nT =nT+max([nnz(H.num{j,k})-1 0])+max([nnz(H.den{j,k})-1 0]);
                    end
                end
                % Find indices for the parameters and construct transfer
                % function objects
                obj.param = oidParameter(zeros(nT,1),1:nT);
                n = 1;
                for j = 1:p
                    for k = 1:m
                        [iNumG n] = obj.paramIndices(G.num{j,k},n,false);
                        [iDenG n] = obj.paramIndices(G.den{j,k},n,true);                      
                        obj.G(j,k) = oidTf(G(j,k),iNumG,iDenG,obj.param);
                    end
                end
                for j = 1:p
                    for k = 1:p
                        [iNumH n] = obj.paramIndices(H.num{j,k},n,true);
                        [iDenH n] = obj.paramIndices(H.den{j,k},n,true);
                        obj.H(j,k) = oidTf(H(j,k),iNumH,iDenH,obj.param);
                    end
                end
            % User specified parameter indices
            elseif numel(varargin) == 4
                % Parameter count
                nT = 0;
                for j = 1:p
                    for k = 1:m
                         nT = max([nT varargin{1}{j,k}(:)' varargin{2}{j,k}(:)']);
                    end
                    for k = 1:p 
                        nT = max([nT varargin{3}{j,k}(:)' varargin{4}{j,k}(:)']);
                    end
                end
                % Use specified indices to construct transfer function
                % objects
                obj.param = oidParameter(zeros(nT,1),1:nT);
                for j = 1:p
                    for k = 1:m
                        [iNumG ~] = obj.paramIndices(...
                            G.num{j,k},varargin{1}{j,k}(:)',false);
                        [iDenG ~] = obj.paramIndices(...
                            G.den{j,k},varargin{2}{j,k}(:)',true);
                        obj.G(j,k) = oidTf(G(j,k),iNumG,iDenG,obj.param);
                    end
                end
                for j = 1:p
                    for k = 1:p
                        [iNumH ~] = obj.paramIndices(...
                            H.num{j,k},varargin{3}{j,k}(:)',true);
                        [iDenH ~] = obj.paramIndices(...
                            H.den{j,k},varargin{4}{j,k}(:)',true);
                        obj.H(j,k) = oidTf(H(j,k),iNumH,iDenH,obj.param);
                    end
                end
                
            else
                throw(MException(...
                    '','MOOSE::oidModelTF::Wrong number of input arguments'));
            end
        end
        
        function n = get.n(obj)
            n = numel(obj.param);
        end
        function theta0 = get.theta0(obj)
            theta0 = zeros(numel(obj.param),1);
            for j = 1:numel(obj.param)
                theta0(j) = obj.param(j).nominalValue;
            end
        end
        
        [G H] = tf(obj);
        dG = dG(obj,n);
        dH = dH(obj,n);
        bode(obj);
        bodemag(obj);
    end
    
    methods (Access = private)
        [I n] = paramIndices(~,coeff,n,isMonic);
    end
end