% Gaussian to non-Gaussian sparse data (Unimodal to Two-modal)
% Sparse data: Where the non-spase data takes value from Unimodal Gaussian
% to Two-modal Non-Gaussian (mixture of two Gaussians)

% For sparse data creation where the data is approximately sparse
% Data follows: X=U+V. Here, 'U' is K-sparse and 'V' is (N-K)-sparse
% The non-zero coeffs of U takes values from (K/2)*N(-a,sigma_2)+(K/2)*N(a,sigma_2)
% The non-zero coeffs of V takes values from N(0,gamma_2)
% Generally sigma_2 > gamma_2. That means that 'U' is predominant over 'V'.
% However, the above relation depends on the value of 'a'

% Written by: Saikat Chatterjee of KTH
% Email: saikatchatt@gmail.com
% Created: 5'th August 2010


function [X U] = SparseDataCreate_G_to_NonG(NoOfData,N,K,sigma_2,gamma_2,a)

% Input
% NoOfData -> Number of vectors to be generated
% N -> Dimension of the approximately sparse vector X
% K -> Zero-norm of U. So, X is nearly K-sparse. K should be an even number  
% sigma_2 -> Variance of the each mixture for the non-zero components of 'U' (two Gaussian mixtures)
% gamma_2 -> Variance of the non-zero components of 'V'
% a -> absolute value of the mean of each mixture. Mean can take either 'a' or '-a' 

% Output
% The approximately sparse data 'X'
% The strictly sparse data 'U'


% Data X=U+V
X=zeros(NoOfData,N);
U=zeros(NoOfData,N);

% For U

% For one mixture with mean '-a'
Mu_U1_1=ones(1,K/2)*(-a);
Sigma_U1_1=diag(sigma_2*ones(1,K/2));
U1_1=mvnrnd(Mu_U1_1,Sigma_U1_1,NoOfData);

% For Second mixture with mean 'a'
Mu_U1_2=ones(1,K/2)*(a);
Sigma_U1_2=diag(sigma_2*ones(1,K/2));
U1_2=mvnrnd(Mu_U1_2,Sigma_U1_2,NoOfData);

U1=[U1_1 U1_2];



% For V
Mu_V1=zeros(1,N-K);
Sigma_V1=diag(gamma_2*ones(1,N-K));
V1=mvnrnd(Mu_V1,Sigma_V1,NoOfData);


for i=1:NoOfData
    
    x=zeros(1,N);
    q = randperm(N);
    
    x(q(1:K)) = U1(i,:);
    U(i,:)=x;
    
    x(q(K+1:N)) = V1(i,:);
    X(i,:)=x;
end