% Implementing the Subspace Pursuit Algorithm of Wei Dai and Olgica Milenkovic 
% Written following the Algorithm 1 Table of Section III of the authors' paper 
% in IEEE Trans IT
% Paper: Subspace pursuit for compressive sensing signal reconstruction;
% IEEE Trans IT, May 2009

% Signal model: y=AX+W. In the authors' paper, main consideration is 'W' is
% a zero vector (i.e. no measurement noise)

% Written by: Saikat Chatterjee of KTH
% Email: saikatchatt@gmail.com
% Created: 5'th May 2010


function [xp I y_r]=Subspace_pursuit_func(y,A,K,N)

% Input
% N <- Dimension of the input data 'X' that is K-sparse (either strictly or weakly)
% y <- 'M' dimensional measurement data (an instance of Y where Y=AX+W)
% K <- Explicit estimate of the sparsity
% A <- Measument matrix of dimension M*N

% Output
% xp -> estimate of 'X' 
% I -> Support set of xp_SP
% y_r -> residue of the mismatch (please see the paper to understand)



% Initialization
x0 = A'*y;   % Initial guess (min energy)
[I0 I0_compliment]=support_set_eval(x0,K);   % Support set evaluation

y_residue0=residue_for_Subspace_pursuit(y,A,I0);

y_residue_old=y_residue0;
I_Old=I0;

loop_dummy=1;
while (loop_dummy > 0)
    
    I_extended=support_set_eval(A'*y_residue_old,K);  % Finding largest 'K' amplitude coeffs through residue correlation matching
    I_union=union(I_Old,I_extended);
            
    A_I_union=A(:,I_union);
    x_dummy_p=pinv(A_I_union) * y;
    
    x_p=zeros(N,1);
    x_p(I_union)=x_dummy_p;
    
    [I_New I_New_compliment]=support_set_eval(x_p,K);
    y_residue_new=residue_for_Subspace_pursuit(y,A,I_New);
    
    if ( norm(y_residue_new) >= norm(y_residue_old) )
        loop_dummy=-1;
        I_New=I_Old;
        y_residue_new=y_residue_old;
    else
        I_Old=I_New;
        y_residue_old=y_residue_new;
    end
end

A_I_New=A(:,I_New);
xp_high_amplitude=pinv(A_I_New) * y;

xp_Subspace_pursuit=zeros(N,1);
xp_Subspace_pursuit(I_New)=xp_high_amplitude;

xp=xp_Subspace_pursuit;
y_r=y_residue_new;
I=I_New;



% -------------------------------------------------------
function y_residue=residue_for_Subspace_pursuit(y,A,I)

% y <- measurement
% A <- CS measurement matrix
% I <- Support set for large amplitude coeffs

% y_residue -> orthogonal projection residue


% Forming a matrix by picking columns indexed by I
A_I=A(:,I);
y_p=A_I * pinv(A_I) * y;
y_residue = y - y_p;
% -------------------------------------------------------