function [rho,aspeed,T,p,mu] = isaatm(altm, Tplus)
% [rho,aspeed,T,p,mu] = isaatm(altm, Tplus)
%
% Compute atmospheric properties for the 1976 standard 
% atmosphere up to 51 000 m. 
%
% altm :   geometric altitude [m]
% Tplus :  (optional) ground temperature difference [K]
%          example: use 15.0 for ISA +15 conditions
% rho :    air density [kg/m^3]
% aspeed : speed of sound [m/s]
% T :      temperature [K]  
% p :      pressure [p]
% mu :     dynamic or absolute viscosity (mu or eta) [Pa s]

  % default is ISA+0
  if nargin < 2
    Tplus = 0.0;
  end

  % constants
  g = 9.80665;
  radius = 6356766.0;
  R = 287.0;
  gamma = 1.4;
  
  % constants for Sutherland's law
  Tmo = 291.15;
  C = 120.0;
  muo = 18.27e-6;
  
  % Compute geopotential altitude
  h = altm*radius / (radius+altm);
  
  % reference conditions at sea level
  L0 = -6.5e-3;
  T0 = 288.16 + Tplus;
  p0 = 1.01325e5;

  % reference conditions at 11 000m
  T11 = T0 + 11000*L0;
  p11 = p0 * (T11/T0)^(-g/(L0*R));

  % lower stratosphere reference conditions
  L20 = +1.0e-3;
  p20 = p11 * exp( -g*(9000.0/(R*T11)) );

  % upper stratosphere reference conditions
  L32 = +2.8e-3;
  T32 = T11 + L20*12000.0;
  p32 = p20 * (T32/T11)^(-g/(L20*R));

  % stratopause reference conditions
  T47 = T32 + L32*15000.0;
  p47 = p32 * (T47/T32)^(-g/(L32*R));
  
  if h < 11000.0
     
    % troposphere: linear temperature gradient
    ho = 0;
    T = T0 + L0*(h - ho);
    p = p0 * (T/T0)^(-g/(L0*R));
  
  elseif h < 20000.0
    
    % tropopause: constant temperature
    ho = 11000.0;
    T = T11;
    p = p11 * exp( -g*(h - ho)/(R*T) );
  
  elseif h < 32000.0 
    
    % lower stratosphere
    ho = 20000.0;
    T = T11 + L20*(h - ho);
    p = p20 * (T/T11)^(-g/(L20*R));
    
  elseif h < 47000.0
  
    % upper stratosphere
    ho = 32000;
    T = T32 + L32*(h - ho);
    p = p32 * (T/T32)^(-g/(L32*R));
    
  elseif h < 51000.0  
    
    % stratopause
    ho = 47000;
    T = T47;
    p = p47 * exp( -g*(h - ho)/(R*T) );
    
  else
  
    error('Standard atmosphere defined up to 51km altitude.');
    
  end

  % derived properties
  rho = p/(R*T);
  aspeed = sqrt(gamma*R*T);
  mu = muo * (Tmo + C)/(T + C) * (T/Tmo)^1.5;

