%
% main program for lift/drag computation example
%
clear all;

% some more aircraft data for fuel efficiency plots
nseat = 110;           % passenger capacity
cabinload = 0.82;      % cabin load factor
fuel_density = 0.804;  % kg/L, jet fuel

% 
% compute drag and thrust for a grid of Mach numbers and altitudes
%
nmach = 30;
nalt = 13;
vmach = linspace(0.35, 0.93, nmach);
valt = linspace(5000, 17000, nalt);

% make some space for results
D = zeros(nmach, nalt);
T = zeros(nmach, nalt);
FC = zeros(nmach, nalt);

% load engine data from file
em = build_engine_model('BR725MachAltMap.txt');

% loop over mach and altitude
for ki = 1:nmach
    for kj = 1:nalt
        D(ki,kj) = drag(vmach(ki), valt(kj));
        [T(ki,kj), FC(ki,kj)] = engine(em, vmach(ki), valt(kj));
    end
end

% we have two engines!
T = 2*T;
FC = 2*FC;

%
% plot thrust and drag as a function of Mach number
%
figure(1);

% Select altitude at which to plot thrust and drag.
% The first altitude in valt (index 1) is 5000m, so 
% (altitude in km) - 4 is the index we want.
icol = 12 - 4;

plot( vmach, D(:,icol)*1e-3, 'r-', vmach, T(:,icol)*1e-3, 'b-');
legend('Drag', 'Thrust', 0);
title(['Drag and thrust at ' num2str(valt(icol)), 'm altitude']);
xlabel('Looks like you forgot this one, too');
ylabel('Yeah now, what is this? ');

% TODO
% remove this once you got the above to work 
return;

%
% compute fuel consumption in L/(100 pax km)
%
npax = nseat * cabinload;
psfc = zeros(size(D));
for ki = 1:nmach
    for kj = 1:nalt
        [rho,aspeed] = isaatm(valt(kj));
        speed = vmach(ki) * aspeed;            % m/s
        
        % fraction of the maximum available engine thrust that we need
        throttle = D(ki,kj) / T(ki,kj);   
        
        % if D/T > 1, then we have more drag than thrust; if D is NaN,
        % we can't even fly there because CL > CL_MAX 
        if isnan(throttle) || throttle > 1
            psfc(ki,kj) = NaN;
        else
            % TODO:
            % Compute the energy efficiency in terms of liters of jet fuel
            % burned per passsenger and 100km flight
            % We have 'npax' passengers and fly with 'speed' meters/second  
            fuel_burn = throttle*FC(ki,kj);  % this is in kg/second
            psfc(ki,kj) = replace_me();
        end
    end    
end

%
% plot fuel consumption graph
%
figure(2);
jcol = [6 9 12] - 4;
plot( vmach, psfc(:,jcol(1)), 'o-', vmach, psfc(:,jcol(2)), 'x-', ...
      vmach, psfc(:,jcol(3)), 'v-'); 
xlabel('Cruise Mach number');
ylabel('Specific fuel consumption [L/100pkm]');
legend( ['Altitude ' num2str(valt(jcol(1))) 'm'], ...
        ['Altitude ' num2str(valt(jcol(2))) 'm'], ...
        ['Altitude ' num2str(valt(jcol(3))) 'm'] );