<?php

namespace Chat\Integration;

use Chat\Model\Entry;

/**
 * A facade for the conversation data store. Manages all create/read/update/delete operations on 
 * entries in the conversation.
 */
class ConversationStore {

    const FILE_NAME = 'conversation.txt';
    const PATH_TO_APP_ROOT = '/../../../';
    const ENTRY_DELIMITER = ";\n";

    private $file_path;

    /**
     * Creates a new instance.
     */
    public function __construct() {
        $this->file_path = __DIR__ . self::PATH_TO_APP_ROOT . self::FILE_NAME;
    }

    /**
     * Appends a new entry to the current conversation.
     * 
     * @param \Chat\Model\Entry $entry The entry to append.
     */
    public function addEntry(Entry $entry) {
        \file_put_contents($this->file_path, \serialize($entry) . self::ENTRY_DELIMITER, FILE_APPEND);
    }

    private function examineConversation($entry_handler) {
        $entry_strings = \explode(self::ENTRY_DELIMITER, \file_get_contents($this->file_path));
        $entries = array();
        foreach ($entry_strings as $entry_string) {
            $entry = \unserialize($entry_string);
            if ($entry instanceof Entry) {
                $entry_handler($entries, $entry);
            }
        }
        return $entries;
    }

    /**
     * Delete the entry with the specified timestamp.
     * 
     * @param int $timestamp The timestamp of the entry that shall be deleted.
     */
    public function deleteEntry($timestamp) {
        $file_path = $this->file_path;
        $entry_delimiter = self::ENTRY_DELIMITER;
        $this->examineConversation(function(array &$entries, Entry $entry) use ($timestamp,
                $file_path,
                $entry_delimiter) {
            if ($entry->getTimestamp() === $timestamp) {
                $entry->setDeleted(true);
            }
            \array_push($entries, serialize($entry));
            file_put_contents($file_path, implode($entry_delimiter, $entries) . $entry_delimiter);
        });
    }

    /**
     * @param boolean $removeDeleted If true, the returned array will not include delted entries.
     * @return array The entire conversation as an array of Entry <code>objects</code>. If the
     *                conversation is empty, also the returned array is empty.
     */
    public function getConversation($removeDeleted) {
        return $this->examineConversation(function(array &$entries, Entry $entry)
                        use ($removeDeleted) {
                    if (!$removeDeleted || !$entry->isDeleted()) {
                        \array_unshift($entries, $entry);
                    }
                });
    }

    /**
     * @param integer $lastReadTime The time when entries was last read.
     * @param boolean $removeDeleted If true, the returned array will not include delted entries.
     * @return array All entries newer than <code>last_read_time</code> as an array of 
     *                Entry <code>objects</code>.
     */
    public function getNewerEntries($lastReadTime, $removeDeleted) {
        return $this->examineConversation(function(array &$entries, Entry $entry) 
                use ($lastReadTime, $removeDeleted) {
                    if ($entry->getTimestamp() > $lastReadTime &&
                            (!$removeDeleted || !$entry->isDeleted())) {
                        \array_unshift($entries, $entry);
                    }
                });
    }

}