<?php

namespace Chat\Controller;

use Chat\Integration\ConversationStore;
use Chat\Model\Entry;

/**
 * This is the application's sole controller. All calls from view to lower layers pass through here.
 */
class Controller {

    private $conversation;
    private $username;

    /**
     * Constructs a new instance.
     */
    public function __construct() {
        $this->conversation = new ConversationStore();
    }

    /**
     * Appends a new entry to the current conversation.
     * 
     * @param \Chat\Model\Entry $entry The entry to append.
     */
    public function addEntry(Entry $entry) {
        $this->conversation->addEntry($entry);
    }

    /**
     * @param boolean $removeDeleted If true, the returned array will not include delted entries.
     * @return array                 The conversation, as an array of <code>Entry</code> objects.
     */
    public function getConversation($removeDeleted = false) {
        return $this->conversation->getConversation($removeDeleted);
    }

    /**
     * @param integer $lastReadTime The time when entries was last read.
     * @param boolean $blocking If true, do not return until the conversation is updated 
     *                           after $lastReadTime.
     * @param boolean $removeDeleted If true, the returned array will not include deleted entries.
     * @return array All entries newer than <code>last_read_time</code> as an array of 
     *                Entry <code>objects</code>.
     */
    public function getNewerEntries($lastReadTime, $blocking, $removeDeleted = false) {
        return $this->conversation->getNewerEntries($lastReadTime, $blocking, $removeDeleted);
    }

    /**
     * Login is not implemented, the specified username is simply stored and used to identify
     * the user in the conversation.
     * 
     * @param type $username The user's nick name, which is shown in the conversation.
     */
    public function login($username) {
        $this->username = $username;
    }

    /**
     * @return string The user's username, which is also the display name.
     */
    public function getUsername() {
        return $this->username;
    }

    /**
     * Delete the entry with the specified timestamp.
     * 
     * @param int $timestamp The timestamp of the entry that shall be deleted.
     */
    public function deleteEntry($timestamp) {
        $this->conversation->deleteEntry($timestamp);
    }

}
