<?php

namespace Chat\Integration;

use Chat\Model\Entry;

/**
 * A facade for the conversation data store. Manages all create/read/update/delete operations on 
 * entries in the conversation.
 */
class ConversationStore {

    const FILE_NAME = 'conversation.txt';
    const PATH_TO_APP_ROOT = '/../../../';
    const ENTRY_DELIMITER = ";\n";
    const ONE_SEC = 1;

    private $file_path;

    /**
     * Creates a new instance.
     */
    public function __construct() {
        $this->file_path = __DIR__ . self::PATH_TO_APP_ROOT . self::FILE_NAME;
    }

    /**
     * Appends a new entry to the current conversation.
     * 
     * @param \Chat\Model\Entry $entry The entry to append.
     */
    public function addEntry(Entry $entry) {
        \file_put_contents($this->file_path, \serialize($entry) . self::ENTRY_DELIMITER, FILE_APPEND);
    }

    private function examineConversation($entry_handler) {
        $entry_strings = \explode(self::ENTRY_DELIMITER, \file_get_contents($this->file_path));
        $entries = array();
        foreach ($entry_strings as $entry_string) {
            $entry = \unserialize($entry_string);
            if ($entry instanceof Entry) {
                $entry_handler($entries, $entry);
            }
        }
        return $entries;
    }

    /**
     * Delete the entry with the specified timestamp.
     * 
     * @param int $timestamp The timestamp of the entry that shall be deleted.
     */
    public function deleteEntry($timestamp) {
        $file_path = $this->file_path;
        $entry_delimiter = self::ENTRY_DELIMITER;
        $this->examineConversation(function(array &$entries, Entry $entry) use ($timestamp,
                $file_path,
                $entry_delimiter) {
            if ($entry->getTimestamp() === $timestamp) {
                $entry->setDeleted(true);
            }
            \array_push($entries, serialize($entry));
            file_put_contents($file_path, implode($entry_delimiter, $entries) . $entry_delimiter);
        });
    }

    /**
     * @param boolean $removeDeleted If true, the returned array will not include delted entries.
     * @return array The entire conversation as an array of Entry <code>objects</code>. If the
     *                conversation is empty, also the returned array is empty.
     */
    public function getConversation($removeDeleted) {
        return $this->examineConversation(function(array &$entries, Entry $entry)
                        use ($removeDeleted) {
                    if (!$removeDeleted || !$entry->isDeleted()) {
                        \array_unshift($entries, $entry);
                    }
                });
    }

    /**
     * @param integer $lastReadTime The time when entries was last read.
     * @param boolean $blocking If true, do not return until the conversation is updated 
     *                           after $lastReadTime.
     * @param boolean $removeDeleted If true, the returned array will not include delted entries.
     * @return array All entries newer than <code>last_read_time</code> as an array of 
     *                Entry <code>objects</code>.
     */
    public function getNewerEntries($lastReadTime, $blocking, $removeDeleted) {
        if ($blocking) {
            $this->waitForNewEntry($lastReadTime);
        }
        return $this->examineConversation(function(array &$entries, Entry $entry)
                        use ($lastReadTime, $removeDeleted) {
                    if ($entry->getTimestamp() > $lastReadTime &&
                            (!$removeDeleted || !$entry->isDeleted())) {
                        \array_unshift($entries, $entry);
                    }
                });
    }

    /**
     * Blocks until there are entries newer than $last_read_time. The session store is closed while
     * sleeping to allow other requests to access session variables.
     * 
     * @param int $lastReadTime The last time an entry was read. Entries with timestamps
     *                              bigger than this have not been read.
     */
    private function waitForNewEntry($lastReadTime) {
        while ($this->lastUpdateTime() <= $lastReadTime) {
            \session_write_close();
            sleep(self::ONE_SEC);
            \session_start();
        }
    }

    private function lastUpdateTime() {
        \clearstatcache();
        return \filemtime($this->file_path);
    }

}