@doc """`pole(sys)`

Compute the poles of system `sys`.""" ->
pole(sys::StateSpace) = eig(sys.A)[1]
pole(sys::TransferFunction) = vcat(map(pole, sys.matrix)...)
pole(sys::SisoTf) = roots(sys.den)
pole(sys::ZPK) = vcat(sys.p...)

@doc """`gain(sys)`

Compute the gain of SISO system `sys`.""" ->
function gain(sys::StateSpace, zs::Vector=tzero(sys))
    !issiso(sys) && error("Gain only defined for siso systems")
    nx = sys.nx
    nz = length(zs)
    return nz == nx ? sys.D[1] : (sys.C*(sys.A^(nx - nz - 1))*sys.B)[1]
end
function gain(sys::TransferFunction)
    sisocheck(sys, "Gain")
    s = sys.matrix[1, 1]
    return s.num[1]/s.den[1]
end
function gain(sys::ZPK)
    sisocheck(sys, "Gain")
    return sys.k[1]
end

@doc """`markovparam(sys, n)`

Compute the `n`th markov parameter of state-space system `sys`. This is defined
as the following:

`h(0) = D`

`h(n) = C*A^(n-1)*B`""" ->
function markovparam(sys::StateSpace, n::Integer)
    n < 0 && error("n must be >= 0")
    return n == 0 ? sys.D : sys.C * sys.A^(n-1) * sys.B
end

@doc """`z, p, k = zpkdata(sys)`

Compute the zeros, poles, and gains of system `sys`.

### Returns
`z` : Matrix{Vector{Complex128}}, (ny x nu)

`p` : Matrix{Vector{Complex128}}, (ny x nu)

`k` : Matrix{Float64}, (ny x nu)""" ->
function zpkdata(sys::LTISystem)
    ny, nu = size(sys)
    zs = Array(Vector{Complex128}, ny, nu)
    ps = Array(Vector{Complex128}, ny, nu)
    ks = Array(Float64, ny, nu)

    for j = 1:nu
        for i = 1:ny
            zs[i, j], ps[i, j], ks[i, j] = _zpk_kern(sys, i, j)
        end
    end
    return zs, ps, ks
end
zpkdata(sys::ZPK) = sys.z, sys.p, sys.k

function _zpk_kern(sys::StateSpace, iy::Int, iu::Int)
    # sys.B[:,iu] has the format Array{Float64,1} (a 2 is required)

    # This function doesn't seem to work properly (changes the eigenvalues of A)
    #A, B, C = struct_ctrb_obsv(sys.A, sys.B[:, iu], sys.C[iy, :])

    # Do this to skip struct_ctrb_obsv:
    A, B, C = sys.A, sys.B[:, iu:iu], sys.C[iy:iy, :]

    # iy:iy D a matrix instead of float
    D = sys.D[iy:iy, iu:iu]

    # added following line
    B = float64mat(B)
    z = tzero(A, B, C, D)
    nx = size(A, 1)
    nz = length(z)
    k = nz == nx ? D[1] : (C*(A^(nx - nz - 1))*B)[1]
    return z, eigvals(A), k
end
function _zpk_kern(sys::TransferFunction, iy::Int, iu::Int)
    s = sys.matrix[iy, iu]
    return roots(s.num), roots(s.den), s.num[1]/s.den[1]
end

@doc """`Wn, zeta, ps = damp(sys)`

Compute the natural frequencies, `Wn`, and damping ratios, `zeta`, of the
poles, `ps`, of `sys`""" ->
function damp(sys::LTISystem)
    ps = pole(sys)
    if !iscontinuous(sys)
        Ts = sys.Ts == -1 ? 1 : sys.Ts
        ps = log(ps)/Ts
    end
    Wn = abs(ps)
    order = sortperm(Wn)
    Wn = Wn[order]
    ps = ps[order]
    zeta = -cos(angle(ps))
    return Wn, zeta, ps
end

@doc """`dampreport(sys)`

Display a report of the poles, damping ratio, natural frequency, and time
constant of the system `sys`""" ->
# TO DO: @printf currently throws an error if there are complex poles...
function dampreport(io::IO, sys::LTISystem)
    Wn, zeta, ps = damp(sys)
    t_const = 1./(Wn.*zeta)
    header =
    ("|     Pole      |   Damping     |   Frequency   | Time Constant |\n"*
     "|               |    Ratio      |   (rad/sec)   |     (sec)     |\n"*
     "+---------------+---------------+---------------+---------------+")
    println(io, header)
    for i=1:length(ps)
        p, z, w, t = ps[i], zeta[i], Wn[i], t_const[i]
        @printf(io, "|  %-13.3e|  %-13.3e|  %-13.3e|  %-13.3e|\n", p, z, w, t)
    end
end
dampreport(sys::LTISystem) = dampreport(STDOUT, sys)


@doc """`tzero(sys)`

Compute the invariant zeros of the system `sys`. If `sys` is a minimal
realization, these are also the transmission zeros.""" ->
function tzero(sys::TransferFunction)
    if issiso(sys)
        return roots(sys.matrix[1,1].num)
    else
        return tzero(ss(sys))
    end
end
tzero(sys::ZPK) = issiso(sys) ? sys.z[1] : tzero(ss(sys))

# Implements the algorithm described in:

# Emami-Naeini, A. and P. Van Dooren, "Computation of Zeros of Linear
# Multivariable Systems," Automatica, 18 (1982), pp. 415–430.
#
# Note that this returns either Vector{Complex64} or Vector{Float64}
tzero(sys::StateSpace) = tzero(sys.A, sys.B, sys.C, sys.D)
function tzero(A::Matrix{Float64}, B::Matrix{Float64}, C::Matrix{Float64},
        D::Matrix{Float64})
    # Balance the system
    A, B, C = balance_statespace(A, B, C)

    # Compute a good tolerance
    meps = 10*eps()*norm([A B; C D])
    A, B, C, D = reduce_sys(A, B, C, D, meps)
    A, B, C, D = reduce_sys(A', C', B', D', meps)
    if isempty(A)   return Float64[]    end

    # Compress cols of [C D] to [0 Df]
    mat = [C D]
    # To ensure type-stability, we have to annote the type here, as qrfact
    # returns many different types.
    W = full(qrfact(mat')[:Q], thin=false)::Matrix{Float64}
    W = flipdim(W,2) # old: fliplr(W) (changed to stop warning)
    mat = mat*W
    if fastrank(mat', meps) > 0
        nf = size(A, 1)
        m = size(D, 2)
        Af = ([A B] * W)[1:nf, 1:nf]
        Bf = ([eye(nf) zeros(nf, m)] * W)[1:nf, 1:nf]
        zs = eig(Af, Bf)[1]
    else
        zs = Float64[]
    end
    return zs
end

# Implements REDUCE in the Emami-Naeini & Van Dooren paper. Returns transformed
# A, B, C, D matrices. These are empty if there are no zeros.
function reduce_sys(A::Matrix{Float64}, B::Matrix{Float64}, C::Matrix{Float64},
        D::Matrix{Float64}, meps::Float64)
    Cbar, Dbar = C, D
    if isempty(A)
        return A, B, C, D
    end
    while true
        # Compress rows of D
        U = full(qrfact(D)[:Q], thin=false)::Matrix{Float64}
        D = U'*D
        C = U'*C
        sigma = fastrank(D, meps)
        Cbar = C[1:sigma, :]
        Dbar = D[1:sigma, :]
        Ctilde = C[(1 + sigma):end, :]
        if sigma == size(D, 1)
            break
        end

        # Compress columns of Ctilde
        V = full(qrfact(Ctilde')[:Q], thin=false)::Matrix{Float64}
        V = flipdim(V,2) # fliplr(V)
        Sj = Ctilde*V
        rho = fastrank(Sj', meps)
        nu = size(Sj, 2) - rho

        if rho == 0
            break
        elseif nu == 0
            # System has no zeros, return empty matrices
            A = B = Cbar = Dbar = Float64[]
            break
        end
        # Update System
        n, m = size(B)
        Vm = [V zeros(n, m); zeros(m, n) eye(m)]
        if sigma > 0
            M = [A B; Cbar Dbar]
            Vs = [V' zeros(n, sigma) ; zeros(sigma, n) eye(sigma)]
        else
            M = [A B]
            Vs = V'
        end
        sigma, rho, nu
        M = Vs * M * Vm
        A = M[1:nu, 1:nu]
        B = M[1:nu, (nu + rho + 1):end]
        C = M[(nu + 1):end, 1:nu]
        D = M[(nu + 1):end,  (nu + rho + 1):end]
    end
    return A, B, Cbar, Dbar
end

# Determine the number of non-zero rows, with meps as a tolerance. For an
# upper-triangular matrix, this is a good proxy for determining the row-rank.
function fastrank(A::Matrix{Float64}, meps::Float64)
    n, m = size(A, 1, 2)
    if n*m == 0     return 0    end
    norms = Array(Float64, n)
    for i = 1:n
        norms[i] = norm(A[i, :])
    end
    mrank = sum(norms .> meps)
    return mrank
end


@doc """`g = dcgain(sys)`

Compute the DC gain of system `sys`. `g[i,j]` contains the DC gain from input i to output j""" ->
function dcgain(sys::StateSpace)
    A, B, C, D = ssdata(sys)
    if det(A) != 0
        return -C*(A\B) + D
    else
        error("A is not invertible, convert to transferfunction") # <-- fix this!
    end
end
function dcgain(sys::TransferFunction)
    nu, ny = size(sys)
    dc_gain = Array(Float64,nu,ny)
    for i = 1:nu, j = 1:ny
        dc_gain[i,j] = dcgain(sys.matrix[i,j])
    end
    return dc_gain
end
function dcgain(sys::SisoTf)
    sysnum = sys.num.a
    sysden = sys.den.a
    if sysden[end] < sqrt(eps(Float64))
        if sysnum[end] < sqrt(eps(Float64))
            return dcgain(SisoTf(sysnum[1:end-1],sysden[1:end-1])) # Cancel s/s
        else
            return Inf
        end
    else
        return sysnum[end]/sysden[end]
    end
end
function dcgain(sys::ZPK)
    ny, nu = size(sys)
    dc_gain = Array(Float64, ny, nu)
    for i=1:ny, j=1:nu
        dc_gain[i,j] = sys.k[i,j]*real(prod(-sys.z[i,j]))/real(prod(-sys.p[i,j]))
    end
    return dc_gain
end
dcgain(k::Number) = k


@doc """`pm, wpm = phasemargin(sys; deg=true/false)`

Compute the phase margin and corresponding unity gain crossover frequency for `LTISystem` `sys` or data from bode-plot `mag`, `phase`, `w`. By definition, unity gain feedback with phase `pm` would make the closed loop system unstable. By default the result is given in radians.""" ->
function phasemargin(sys::TransferFunction; deg::Bool=false)
    # For sys = 1/(s+1)^2*(s+10)^2*(s+100)^4/100^4/10 the output from phasemargin is off by a noticable ammount. Haven't been able to replicate or find the problem, might be because of instabilities in roots() though...

    !issiso(sys) && error("Gain margin only defined for siso systems")

    Z, P = sys.matrix[1].num, sys.matrix[1].den

    # To find w180, first solve |H(jw)| == 1
    rZ, iZ, rP, iP = polysplit(Z)..., polysplit(P)...
    wctmp = roots(rZ^2 + iZ^2 - rP^2 - iP^2)

    # wc must be real and positive
    wc = real(wctmp[(imag(wctmp) .== 0) & (real(wctmp) .>= 0)])
    isempty(wc) && (return NaN, NaN)

    # Evaluate at the valid frequencies and pick the one where the phase is closest to an odd multiple of π
    pm = angle(vec(freqresp(sys, wc)[1])) # note: -π < pm < π
    ipm = findmax(abs(pm))[2]

    return (pm[ipm]<0 ? π + pm[ipm] : pm[ipm] - π)*(deg ? 360/2π : 1), wc[ipm]
end
# For other representations, convert to transferfunction
phasemargin(sys::LTISystem; deg::Bool=false) = phasemargin(tf(sys), deg=deg)

@doc """`gm, wgm = gainmargin(sys; dB=true/false)`

Compute the gain margin and corresponding 180 degree phase crossing for `LTISystem` `sys`. The gain margin is the smallest ammount of (pure gain) negative feedback that would make the closed loop system unstable. By defrault the result is given in linear units (not dB).""" ->
function gainmargin(sys::TransferFunction; dB::Bool=false)
    !issiso(sys) && error("Gain margin only defined for siso systems")

    Z, P = sys.matrix[1].num, sys.matrix[1].den

    # To find w180, first solve imag(sys(jw)) == 0
    rZ, iZ, rP, iP = polysplit(Z)..., polysplit(P)...
    w180tmp = roots(iZ*rP - rZ*iP)

    # Evaluate the system at those frequencies
    gtmp = real(vec(freqresp(sys, real(w180tmp))[1]))

    # Find valid indexes: w180 must be real and positive, and the phase must be π
    i180 = (imag(w180tmp) .== 0) & (real(w180tmp) .>= 0) & (gtmp .< 0)
    !any(i180) && (return NaN, NaN)

    gm, w180 = abs(gtmp[i180]), real(w180tmp[i180])

    # Find the smallest ammount of gain that would make the system unstable
    igm = findmin(abs(log10(gm)))[2]
    return (dB ? -20*log10(gm[igm]) : 1/real(gm[igm])), w180[igm]
end
# For other representations, convert to transferfunction
gainmargin(sys::LTISystem, dB::Bool=false) = gainmargin(tf(sys), dB=dB)
