# Allows max() to be used on a number
Base.max(x::Number) = x

@doc """`p, k = rlocus(sys, k=...)`

Calculate the root locus for `LTISystem` `sys`. A gain vector `k` can optionally be provided.""" ->
function rlocus(sys::TransferFunction, kvec::AbstractVector{Float64}=[-1.])
    !issiso(sys) && error("rlocus only defined for siso systems")

    # If the system is improper, calculate the locus of 1/sys and reverse pole/zero locations
    if !isproper(sys)
        if kvec != [-1.]
            pmat, kvec = rlocus(1/sys, 1./kvec[end:-1:1])
            return pmat[end:-1:1,:], kvec
        else
            pmat, kvec = rlocus(1/sys, kvec)
            return pmat[end:-1:1,:], 1./kvec[end:-1:1]
        end
    end

    # Generate zero and pole polynomials
    Z, P = sys.matrix[1].num, sys.matrix[1].den

    # If a gain vector is provided manually, use that one
    kvec != [-1.] && (return rlsort(generatepoles(Z, P, kvec)), kvec)

    # Otherwise generate one that produces a smooth plot:

    # Calculate locus properties:
    kmult = rlbreakaway(Z, P)
    asymptotes, midpoint = rlasymptotes(Z, P)

    # Generate initial gain vector (of length 51)
    kinit = rlgain(Z, P, kmult, asymptotes)

    # Generate a smoother locus where necessary
    kvec, pmat = smoothloc(Z, P, kinit, kmult, asymptotes, midpoint)

    return pmat, kvec
end
rlocus(sys::LTISystem, kvec::AbstractVector) = rlocus(tf(sys), map(Float64, kvec))
rlocus(sys::LTISystem; k::AbstractVector = [-1.]) = rlocus(tf(sys), map(Float64, k))

# Calculates the asymptotes of the system with transferfunction Z/P and the point where they intersect (on the real axis)
function rlasymptotes(Z::Poly{Float64}, P::Poly{Float64})
    np, nz = length(P)-1, length(Z)-1
    m = np-nz
    m<0 && (return rlasymptotes(P,Z))

    if m != 0
        midpoint = -((np>=1 ? P[2]/P[1] : 0.0) - (nz>=1 ? Z[2]/Z[1] : 0.0))/m
        if Z[1]/P[1] > 0
            asymptotes = collect(2*(0:m-1)+1)*pi/m
        else
            asymptotes = collect(2*(0:m-1))*pi/m
        end
        return asymptotes, midpoint
    else
        return Array(Float64,0), NaN
    end
end


# Determine an initial set of gain values for the root locus
function rlgain(
    Z::Poly{Float64},
    P::Poly{Float64},
    kmult::Vector{Float64},
    asymptotes::Vector{Float64}
    )
    kstart = rlgainstart(Z, P)
    ktmp = 1/rlgainstart(P, Z)
    kend = isempty(kmult) ? ktmp : max(ktmp, 2*kmult[end])
    if length(asymptotes) > 0
      kasym = rlasymptotestart(Z, P)
      kmax = max(kend, 2*kmult..., kasym...)
      kmin = kstart
    else
      # No asymptotes
      kmin = kstart
      kmax = max(kend, kmin+1)
    end

    return [0; logspace(log10(kmin), log10(kmax), 50)]
end

# Calculated a value of K for which the root locus of sys is peturbed by no more than 10 %.
function rlgainstart(
    Z::Poly{Float64},
    P::Poly{Float64},
    )
    thresh = 0.1

    z, p, k = roots(Z), roots(P), Z[1]/P[1]
    nz, np = length(z), length(p)
    m = abs(np-nz)
    asymptotes = collect(2*(0:m-1)+1)*pi/(m==0 ? 1 : m)  # Asymptote angles
    isatzero = map(x->x<1e3*eps(Float64), abs(p))
    ds = thresh*abs(p) + isatzero*max(eps(Float64), thresh^sum(isatzero))

    zradius = maxabs([1; 2*z])
    s = [p + ds.*exp(100im*angle(p));
            zradius*exp(im*asymptotes)]

    dk = [polyval(P, s[i])/polyval(Z, s[i]) for i=1:m+np]
    dkmin = minabs(dk)
    return max(dkmin, 10.0^-60)
end

function rlasymptotestart(Z::Poly{Float64}, P::Poly{Float64})
  z, p = roots(Z), roots(P)
  wmax = max(abs(z)...,abs(p)...)
  wmax == 0 && (wmat = 1)
  m = length(P) - length(Z)
  k = Z[1]/P[1]

  # use that H(s) ~ k/s^m for |s|>>wmax
  return (3*wmax)^m/k
end

# Returns vector of real gains for which the locus breaks away from the real line
function rlbreakaway(Z::Poly{Float64}, P::Poly{Float64})
    # Solve P'Z-Z'P = 0
    pmult = roots(polyder(P)*Z - P*polyder(Z))
    n = length(pmult)
    kmult = Array(Float64, 0)
    for i=1:n
        ktmp = -polyval(P,pmult[i])/polyval(Z,pmult[i])
        if isreal(ktmp) && real(ktmp)>=0. && !in(real(ktmp), kmult)
          kmult = [kmult; real(ktmp)]
        end
    end
    return sort(kmult)
end

function smoothloc(
    Z::Poly{Float64},
    P::Poly{Float64},
    kinit::Vector{Float64},
    kmult::Vector{Float64},
    asymptotes::Vector{Float64},
    midpoint::Float64
    )
    # Tolerance for asymptote tracking
    m = length(P) - length(Z)
    asytol = pi/60
    kinit = sort([kinit; 0.999*kmult ;kmult; 0.001*kmult])
    kend = kinit[end]

    # Calculate locus pole locations for initial gain vector and limit
    pinit = generatepoles(Z, P, kinit)
    pinf = limitzeros(Z, P, vec(pinit[end-1,:]))

    # estimate scale of locus
    scale = maximum(abs(pinit[:,1]))

    # Maximum number of iterations
    maxind = 200
    # Note: since leftover gain values from kinit are also inserted the actual maximum array size for k is 200+50+3*length(kmult)+1

    # Allocate arrays to hold gain values and poles
    np = size(pinit,2)
    k = zeros(Float64, maxind)
    p = zeros(Complex128, maxind, np)
    k[1:2] = kinit[1:2]
    p[1:2,:] = rlsort(pinit[1:2,:])

    # Define indexes
    i = 3 # location in new smooth gain vector
    j = 3 # location in initial gain vector

    knext = 0.
    done = false

    # Loop until a smooth locus is achieved or until the gain vector is filled (and make sure the gain isn't unreasonably large)
    while (!done) && (i<maxind) && (knext<10.0^60)
        knext = kinit[j]
        pinit[j,:] = pnext = rlsort(vec(p[i-1,:]), vec(pinit[j,:]))

        # might want to check || in(k[i-1], kmult)
        # Check if the paths joining the last sets of poles is smooth. Skip if the difference in k is too small (prevents the loop from trying to smooth out numerical errors).
        if  knext < 1.001*k[i-1] || issmooth(vec(p[i-2,:]), vec(p[i-1,:]), vec(pnext), scale) || knext - k[i-1] < 1e-6
            # knext is accepted
            k[i] = knext
            p[i,:] = pnext
            scale = max(scale, maxabs(pnext))

            # If the gain is large enough, check if transition to endpoints is smooth
            if knext >= kend
                # resort pinf
                pinf = rlsort(vec(p[i,:]), pinf)
                # Find indexes for which the limit poles are finite or asymptotically infinite
                ifin, iasy = find(isfinite, pinf), find(isinf, pinf)
                # Check that transition to limit poles is smooth
                done = issmooth(vec(p[i-1,ifin]), vec(p[i,ifin]), vec(pinf[ifin]), scale)

                # Check that asymptotes are reached if there are any
                if m>0
                    curdirect = mod(angle(p[i, iasy]-p[i-1, iasy]), 2*pi)
                    dirgap = abs(sort(vec(curdirect))-vec(asymptotes))
                    done = done && !any(dirgap .> asytol)
                end
            end
            j += 1
            i += 1
        else # The locus is not smooth, add another point.
            # Let next point be the (logarithmic) average of k[end] and kinit[j]
            if knext > 10*k[i-1]
                midk = sqrt(k[i-1]*knext)
            else
                midk = (k[i-1]+knext)/2
            end
            kinit = [midk; kinit[j:end]]
            pinit = [generatepoles(Z, P, [midk]); pinit[j:end,:]]
            j = 1
        end

        # Generate more points in kint if k[end] doesn't satisfy smoothness conditions
        if j > length(kinit) && !done
            lk = log10(k[i-1])
            kinit = [kinit; logspace(lk+0.1, lk+1.1, 3)]
            pinit = [pinit; generatepoles(Z, P, kinit[end-2:end])]
            kend = kinit[end]
        end
    end # End of while

    # Add in unused gain values from kinit and add k->inf limit
    kout = [k[1:i-1]; kinit[j+1:end]; Inf]
    pout = [p[1:i-1,:]; rlsort(vec(p[i-1,:]), pinit[j+1:end,:]); pinf.']

    # Extend asymptotes if there are any (instead of poles ending with Inf+0im)
    if m > 0
        # Poles go to infinity as k->inf
        iasy = find(isinf, pinf)
        pmax = 100*maxabs(pout[end-1,:])
        for i = 1:m
            pinf[iasy[i]] = midpoint + cos(asymptotes[i])*pmax + im*sin(asymptotes[i])*pmax
        end
    end
    pout[end,:] = rlsort(vec(pout[end-1,:]), pinf)
    return kout, pout
end

function issmooth(p1,p2,p3::Vector{Complex128}, scale)
  # Check that the curves joining the sets or roots p1,p2 and p3 are smooth enough
  if isempty(p1) || isempty(p2)
    return true
  else
    # Definite maximum admissible angle between (p1,p2) and (p2,p3)
    mincos = 0.985
    # Geometric variables
    p12, p23 = p2-p1, p3-p2
    d12 = real(p12).^2 + imag(p12).^2
    d23 = real(p23).^2 + imag(p23).^2
    rho = (real(p12).*real(p23) + imag(p12).*imag(p23))/mincos
    return all( (d23 .< (scale/100)^2) |((rho .>= 0) & (rho.^2 .>= d12.*d23)))
  end
end

function generatepoles(
  Z::Poly{Float64},
  P::Poly{Float64},
  kvec::Vector{Float64},
  )
  nk, np = length(kvec), max(length(Z),length(P))-1

  # Allocate an extra row of NaN for limit zeros if longer == true
  pmat = [NaN+0im for i=1:nk, j=1:np]
  for ik = 1:nk
    p = roots(P + kvec[ik]*Z)
    for ip = 1:length(p)
      pmat[ik, ip] = abs(p[ip]) < 1e-3*realmax(Float64) ? p[ip] : NaN
    end
  end
  return pmat
end

# Sort a matrix of pole locations so that the locus takes as small steps as possible between gain values
function rlsort(pmat::Matrix{Complex128})
    nk, np = size(pmat)
    psort = Array(Complex128, size(pmat))
    psort[1,:] = pmat[1,:]
    for ik=2:nk
      psort[ik, :] = rlsort(vec(psort[ik-1,:]), vec(pmat[ik,:]))
    end
    return psort
end
# A referemce vector for sorting can be provided
function rlsort(p::Vector{Complex128}, pmat::Matrix{Complex128})
    if !isempty(pmat)
      pmat[1,:] = rlsort(p, vec(pmat[1,:]))
      return rlsort(pmat)
    else
    # If pmat is empty: return an empty array
    return Array(Complex128,size(pmat))
    end
end
function rlsort{T<:Number}(p1::Vector{T}, p2::Vector{T})
    np = length(p1)
    psort = Array(Complex128, np)
    avaliable = collect(1:np)

    for i = 1:np
        pavaliable = p1[findin(avaliable, 1:np)]
        # Take care of any poles at infinity (some times caused by the root() function)
        if !isfinite(p2[i])
          pclose, iclose = findmax(abs(pavaliable))
          iclose = pickindex(iclose, avaliable)
          psort[iclose] = Inf
          avaliable[iclose] = 0
        else
          pclose, iclose = findmin(abs(p2[i] - pavaliable))
          iclose = pickindex(iclose, avaliable)
          psort[iclose] = p2[i]

          # Make sure the same index won't be picked twice
          avaliable[iclose] = 0
        end
    end
    return psort
end

# Connect every zero to the closest pole (to simulate lim K->Inf)
function limitzeros(Z::Poly{Float64}, P::Poly{Float64}, pvec::Vector{Complex128})
    np = length(pvec)
    Zroots = roots(Z)
    avaliable = collect(1:np)
    pout = [Inf+0im for i=1:length(pvec)]
    for z in Zroots
        pavaliable = pvec[findin(avaliable, 1:np)]
        if !isempty(pavaliable)
          pclose, iclose = findmin(abs(z - pavaliable))
          iclose = pickindex(iclose, avaliable)
          pout[iclose] = z
          avaliable[iclose] = 0
        else
            return pout
        end
    end
    return pout
end

# Used for keeping track of which poles have been matched in the functions limitzeros and rlsort
function pickindex(i::Int, avaliable::Vector{Int})
    count = 1
    iout = 0
    while count <= i
        avaliable[iout+1] != 0 && (count += 1)
        iout += 1
    end
    return iout
end
