@doc """`sminreal(sys)`

Compute the structurally minimal realization of the state-space system `sys`. A
structurally minimal realization is one where only states that can be
determined to be uncontrollable and unobservable based on the location of 0s in
`sys` are removed.""" ->
function sminreal(sys::StateSpace)
    A, B, C, inds = struct_ctrb_obsv(sys)
    return StateSpace(A, B, C, sys.D, sys.Ts, sys.statenames[inds],
            sys.inputnames, sys.outputnames)
end


# Determine the structurally controllable and observable realization for the system
struct_ctrb_obsv(sys::StateSpace) = struct_ctrb_obsv(sys.A, sys.B, sys.C)
function struct_ctrb_obsv(A::VecOrMat, B::VecOrMat, C::VecOrMat)
    costates = struct_ctrb_states(A, B) & struct_ctrb_states(A', C')
    if !all(costates)
        inds = find(costates)
        return A[inds, inds], B[inds, :], C[:, inds], inds
    else
        return A, B, C, collect(1:size(A, 1))
    end
end

# Compute a bit-vector, expressing whether a state of the pair (A, B) is
# structurally controllable.
function struct_ctrb_states(A::VecOrMat, B::VecOrMat)
    bitA = A .!= 0
    d_cvec = cvec = any(B .!= 0, 2)
    while any(d_cvec .!= 0)
        Adcvec = any(bitA[:, find(d_cvec)], 2)
        cvec = cvec | Adcvec
        d_cvec = Adcvec & ~cvec
    end
    return cvec
end

@doc """`minreal(sys, tol)`

Cancels redundant pole-zero pairs in the LTI system `sys` with tolerance `tol`. If no tolerance is provided it is calculated based on the pole/zero locations.""" ->
function minreal(sys::LTISystem, tol::Float64=NaN)
    zmat, pmat, k = zpkdata(sys)
    ny, nu = size(sys)

    minsys = copy(sys)
    for i=1:ny, j=1:nu
        z, p = minreal(zmat[i,j], pmat[i,j], tol)
        # Build the new TransferFunction if anything was cancelled
        if length(z) < length(zmat[i,j])
            if isa(sys, TransferFunction)
                sisotf = SisoTf(k[i,j]*real(poly(z)), real(poly(p)))
                minsys.matrix[i,j] = sisotf
            else # ZPK model
                minsys.z[i,j] = z
                minsys.p[i,j] = p
            end
        end
    end
    return minsys
end
function minreal(z::Vector{Complex128}, p::Vector{Complex128}, tol::Float64=NaN)
    z, p = copy(z), copy(p)
    nz, np = length(z), length(p)
    icancel = Array(Int, 0)
    calctol = isnan(tol)

    # Loop over zeros
    i = 1
    while i <= length(z)
        zi = z[i]
        # Pick a tolerance if none is provided
        if calctol
            tol = 1000*max(eps(Complex128), abs(zi)*sqrt(eps(Complex128)))
        end

        # Find a cancelling pole if there is one
        icancel= findnext(abs(p-zi) .< tol, 1)

        # Cancel the pole/zero if appropriate
        if icancel != 0
            deleteat!(p, icancel)
            deleteat!(z, i)
        else
            i += 1
        end
    end
    return z, p
end

# TODO: implement minreal for statespace, in the meanwhile convert to transferfunction
minreal(sys::StateSpace) = ss(minreal(tf(sys)))
