@doc """`lqr(A, B, Q, R)`

Calculate the optimal gain matrix `K` for the state-feedback law `u = K*x` that
minimizes the cost function:

J = integral(x'Qx + u'Ru, 0, inf).

For the continuous time model `dx = Ax + Bu`.

`lqr(sys, Q, R)`

Solve the LQR problem for state-space system `sys`. Works for both discrete
and continuous time systems.""" ->
function lqr{T<:BlasFloat}(A::StridedMatrix{T}, B::StridedMatrix{T},
            Q::StridedMatrix{T}, R::StridedMatrix{T})
    S = care(A, B, Q, R)
    K = R\B'*S
    return K
end
lqr{T<:Integer}(A::StridedMatrix{T}, B::StridedMatrix{T}, Q::StridedMatrix{T},
        R::StridedMatrix{T}) = lqr(float(A), float(B), float(Q), float(R))

function lqr(sys::StateSpace, Q, R)
    if iscontinuous(sys)
        return lqr(sys.A, sys.B, Q, R)
    else
        return dlqr(sys.A, sys.B, Q, R)
    end
end

@doc """`dlqr(A, B, Q, R)`, `dlqr(sys, Q, R)`

Calculate the optimal gain matrix `K` for the state-feedback law `u[k] = K*x[k]` that
minimizes the cost function:

J = sum(x'Qx + u'Ru, 0, inf).

For the discrte time model `x[k+1] = Ax[k] + Bu[k]`.""" ->
function dlqr{T<:BlasFloat}(A::StridedMatrix{T}, B::StridedMatrix{T},
            Q::StridedMatrix{T}, R::StridedMatrix{T})
    S = dare(A, B, Q, R)
    K = (B'*S*B + R)\(B'S*A)
    return K
end
dlqr{T<:Integer}(A::StridedMatrix{T}, B::StridedMatrix{T}, Q::StridedMatrix{T},
        R::StridedMatrix{T}) = dlqr(float(A), float(B), float(Q), float(R))

# # Already implemented in matrix_comps
# # Creates controllability matrix
# function ctrb(A::Array, B::VecOrMat)
#     n = size(A, 1)
#     nu = size(B, 2)
#
#     # Check for dimension mismatch
#     n != size(A, 2) && error("A must be square (is $(n)x$(size(A, 2)))")
#     size(A, 2) != size(B, 1) && error("B must have the same row size as A")
#
#     # Allocate controllability matrix
#     co = Array(Float64, n, n*nu)
#     co[:, 1:nu] = B
#     for i=1:n-1
#         co[:, i*nu+1:(i+1)*nu] = A*co[:, (i-1)*nu+1:i*nu]
#     end
#     return co
# end

@doc """`acker(A, B, p)`

Calculates the feedback matrix L such that the siso system given by `dx/dt = Ax + B` with  feedback law `u = -Lx` has closed loop poles at locations specified by the vector `p`.""" ->
function acker{T<:Number}(A::Matrix{Float64}, B::VecOrMat{Float64}, p::Vector{T})
    R = promote_type(T, Float64)

    n = size(A, 1)
    # Check for dimension mismatch
    n != size(A, 2) && error("A must be a square matrix.")
    size(B, 1) != n && error("B must have the same row size as A")
    size(B, 2) != 1 && error("The column size of B must be 1; Pole placement is currently only implmented for siso systems.")

    # Compute the controllability matrix and check the determinant
    CO = ctrb(A, B[:, 1:1])
    det(CO) == 0 && error("System is not controllable; pole placement through the Ackermann method not possible.")

    # Compute the characteristic polynomial
    phi = eye(T, n)
    for i = 1:n
        phi *= A - I*p[i]
    end

    # Place poles using ackermann's method
    L = CO \ phi
    return real(L[end,:])
end
acker(A::Matrix, B::VecOrMat, p::Vector) = acker(map(Float64, A), map(Float64, B), p)
