using DSP

#####################################################################
#=                     IDMethods
            Several System Identification Methods

                Implementation according to

              "Modeling of Dynamical Systems"

                    by Ljung and Glad


Author : Lars Lindemann @2015

                                                                   =#
#####################################################################




#=   Fourier Analysis
For the Fourier Analysis perform a fft and apply
G(jw) = Y(jw)/U(jw)

The fourier transforms of Y and U can be smoothed and averaged

Author : Lars Lindemann @2015
                                                                    =#
@doc """`idSpectralModel = etfe(iddata, [method="pwelch", window="rect", nf=100, smooth=false])`

Performs the estimation of G by the empirical transfer function estimate. As a method, "pwelch" and "single" can be choosen, where
pwelch averages over several chunks of data. As windows, "rect", "hanning", "hamming", "tukey" and "blackman" are available.
The parameter smooth determines, if the estimate is smoothed by a lowpass or not. nf gives the amount of frequencies over which pwelch
averages.

Use for example etfe1 = etfe(iddata, window="hanning");""" ->
function etfe(iddata::iddataObject;method::ASCIIString="pwelch", window::ASCIIString="rect", nf::Int64=100, smooth::Bool=false)
  # Two methods. "Single" calculates ffts of the whole signal, while "chunks"
  # subdivides the signals into different chunks followed by averaging

  fs = 1/iddata.Ts;

  if method=="single"
    N = length(iddata.y)

    # Choose window
    win = selectWindow(window,N);
    # perform fft and build transfer function G
    Y = fft(iddata.y.*win);
    U = fft(iddata.u.*win);

    G = Y./U;

    if smooth
      # now smooth G by a Lowpass
      lowpass        = digitalfilter(Lowpass(0.1),Butterworth(4));
      G_etfe         = filt(lowpass,G);
    else
      G_etfe = G;
    end

    # truncate G
    G_etfe         = flipdim(G_etfe[round(Integer,length(G_etfe)/2):end],1);

    # calculate frequency vector
    f              = collect(0:fs/N:fs-fs/N);
    f              = f[1:length(G_etfe)];

  elseif method=="pwelch"
    # multiply with 2 since half of the frequencies will be disgarded (we are dealing with real signals)
    nf     = nf*2;
    # number of averages possible with the given amount of frequencies
    nr     = floor(Integer,length(iddata.y)/nf);


    # Choose window
    win = selectWindow(window,nf);

    counter        = 0;
    G              = zeros(nf);
    for i=1:nf:nr*nf
      # perform fft and build transfer function G
      if iddata.u[i:i+nf-1]!=zeros(length(iddata.u[i:i+nf-1]))
        Y = fft(iddata.y[i:i+nf-1].*win);
        U = fft(iddata.u[i:i+nf-1].*win);

        G += Y./U;

        counter += 1;
      end
    end

    if smooth
      # lowpass for smoothing
      lowpass        = digitalfilter(Lowpass(0.2),Butterworth(4));
      G_etfe = filt(lowpass,G/counter);
    else
      G_etfe = G/counter;
    end

    N      = length(G_etfe);
    # truncate G_etfe
    G_etfe         = flipdim(G_etfe[round(Integer,length(G_etfe)/2):end],1);

    # calculate frequency vector
    f              = collect(0:fs/N:fs-fs/N);
    f              = f[1:length(G_etfe)];

  end

  idModel   = idSpectralModel(G_etfe,f,"etfe");

  return idModel

end

#=   Spectral Analysis
For the Spectral Analysis calculate the power density spectrum and apply
G(jw) = phi_yu(w)/phi_u(w)

Author : Lars Lindemann @2015
                                                                    =#
@doc """`idSpectralModel = spa(iddata, [method="blackman-tuckey", window="rect", windowLength=30, nf=100])`

Performs the estimation of G by spectral analysis. As a method, "pwelch", "blackman-tuckey" and "single" can be choosen, where
pwelch averages over several chunks of data. As windows, "rect", "hanning", "hamming", "tukey" and "blackman" are available.
nf gives the amount of frequencies over which pwelch averages. windowLength determines the window that is applied in blackman-tukey

Use for example spa1 = spa(sysIdentData, window="hanning");""" ->
function spa(iddata::iddataObject; method::ASCIIString="blackman-tukey",window::ASCIIString="rect",windowLength::Int64=30,nf::Int64=100)

  fs = 1/iddata.Ts;

  if method=="single"
    # Calculate auto- and cross-correlation
    R_yu = normalizeCorr(xcorr(iddata.y,iddata.u),method="biased");
    R_uu = normalizeCorr(xcorr(iddata.u,iddata.u),method="biased");

    # Choose window
    win = selectWindow(window,length(R_yu));

    # perform fft of windowed correlation functions
    N      = length(R_yu);
    phi_yu = fft(R_yu.*win)[1:round(Integer,length(R_yu)/2)];
    phi_uu = fft(R_uu.*win)[1:round(Integer,length(R_uu)/2)];

    # form transfer function and frequency vector
    G = phi_yu./phi_uu;
    f = collect(0:fs/N:fs-fs/N);
    f = f[1:length(G)];

  elseif method=="pwelch"
    # multiply with 2 since half of the frequencies will be disgarded (we are dealing with real signals)
    nf     = nf*2;
    # number of averages possible with the given amount of frequencies
    nr     = floor(Integer,length(iddata.y)/nf);
    # length of correlation chunks
    N      = 2*nf-1;

    # Choose window
    win = selectWindow(window,N);

    counter        = 0;
    phi_yu         = zeros(nf);
    phi_uu         = zeros(nf);
    for i=1:nf:nr*nf
      # calculate correlations for chunks and sum them up
      R_yu = normalizeCorr(xcorr(iddata.y[i:i+nf-1],iddata.u[i:i+nf-1]),method="biased");
      R_uu = normalizeCorr(xcorr(iddata.u[i:i+nf-1],iddata.u[i:i+nf-1]),method="biased");

      # perform fft of windowed correlation functions and already truncate
      phi_yu += fft(R_yu.*win)[1:round(Integer,length(R_yu)/2)];
      phi_uu += fft(R_uu.*win)[1:round(Integer,length(R_uu)/2)];

      counter += 1;
    end
    phi_yu /= counter;
    phi_uu /= counter;

    # Calculate transfer function and frequency vector
    G      = phi_yu./phi_uu;
    f      = collect(0:fs/N:fs-fs/N);
    f      = f[1:length(G)];

  elseif method=="blackman-tukey"

    # Choose window
    win = selectWindow(window,2*windowLength+1);

    # Calculate auto and cross correlation
    Rn_yu    = normalizeCorr(xcorr(iddata.y,iddata.u),method="biased");
    Rn_uu    = normalizeCorr(xcorr(iddata.u,iddata.u),method="biased");

    # disregard not used lags (here: for two sided correlation function)
    if length(Rn_yu)<2*windowLength
      error("Data set too short or too long time window")
    end
    tauZero  = round(Integer,(length(Rn_yu)+1)/2);
    R_yu     = Rn_yu[tauZero-windowLength:tauZero+windowLength];
    R_uu     = Rn_uu[tauZero-windowLength:tauZero+windowLength];

    # multiply window with correlation functions
    z_yu     = R_yu.*win;
    z_uu     = R_uu.*win;

    # perform fft
    N        = length(z_yu);
    phi_yu   = fft(z_yu)[1:windowLength+1];
    phi_uu   = fft(z_uu)[1:windowLength+1];

    # calculate transfer function and frequency vector
    G        = phi_yu./phi_uu;
    f        = collect(0:fs/N:fs-fs/N);
    f        = f[1:length(G)];
  else
    error("Inserted method not available")
  end

  idModel   = idSpectralModel(G,f,"spa");

  return idModel

end

#=   Prediction error method
Prediction error method with the cost function
V = 1/N sum 0.5*(y-y_pred)^2
and step size regulation.

Author : Lars Lindemann @2015
                                                                    =#
@doc """`idLinearModel = PEM(iddata, method, nf, nb, nc, nd, nk [,initC=true])`

Prediction Error Method for method = "arx", "armax", "oe" or "bj" models. nf, nb, nc, nd and nk specify the
order of the polynomials and the input delay respectively. The optional parameter initC determines if initial
values for the Gauß-Newton algorithm should be used.

Use for example arxModelPem = PEM(iddata,"arx",na,nb,0,na,nk);""" ->

function PEM(iddata::iddataObject,method::ASCIIString,nf::Int64,nb::Int64,nc::Int64,nd::Int64,nk::Int64;initC::Bool=true)
  V=0
  # use one of the identification model methods
  if method == "armax"

    if nf!=nd
      error("For the armax model, the f and d polynomial need to be equal and therefore have the same size!")
    elseif nd<0 || nf<0 || nb<1
      error("Order of d, f and b polynomial need to be at least 0, 0 and 1, respectively")
    elseif nc<=0
      error("The order of the c polynomial needs to be greater than zero")
    end

    if initC
      initCond   = init_cond(iddata.u, iddata.y, nf, nb, nc);
      initialX   = [initCond[1][2:nf+1] ; initCond[2][1:nb] ; initCond[3][2:nc+1]];
    else
      initialX   = map(Float64,ones(nf+nb+nc));
    end
    sol          = armax(iddata,initialX,nf,nb,nc,nk);
  elseif method == "arx"

    if nf!=nd
      error("For the arx model, the f and d polynomial need to be equal and therefore have the same size!")
    elseif nc!=0
      error("For the arx model, nc need to be zero")
    end

    initialX   = map(Float64,ones(nf+nb));
    sol        = arx(iddata,initialX,nf,nb,nk);
  elseif method == "oe"

    if nc!=0 || nd!=0
      error("For the output error model, nc and nd need to be zero")
    elseif nb<1 || nf<0
      error("Order of b and f polynomial need to be at least 1 and 0, respectively")
    end

    if initC
      initCond   = init_cond(iddata.u, iddata.y, nf, nb, nc);
      initialX   = [initCond[1][2:nf+1] ; initCond[2][1:nb] ];
    else
      initialX   = map(Float64,ones(nf+nb));
    end

    sol          = oe(iddata,initialX,nf,nb,nk);
  elseif method == "bj"

    if nb<1 || nf<0 || nc<0 || nd<0
      error("Order of b, f, c and d polynomial need to be at least 1, 0, 0 and 0 respectively")
    end

    if initC
      # very inexact ...
      initCond   = init_cond(iddata.u, iddata.y, nf, nb, nc);
      initialX   = [initCond[1][2:nf+1] ; initCond[2][1:nb] ; initCond[3][2:nc+1]; ones(nd)];
    else
      initialX   = map(Float64,ones(nf+nb+nc+nd));
    end

    sol          = bj(iddata,initialX,nf,nb,nc,nd,nk);
  else
    error("No matching method")
  end

  return sol;
end

#=   Prediction error method 2
Prediction error method with the cost function
V = 1/N sum 0.5*(y-y_pred)^2
and step size regulation. This method uses the package ForwardDiff to calculate derivatives of V.

Author : Lars Lindemann @2015
                                                                    =#
@doc """`idLinearModel = PEM2(iddata, method, nf, nb, nc, nd, nk [,initC=true])`

Prediction Error Method for method = "arx", "armax", "oe" or "bj" models. nf, nb, nc, nd and nk specify the
order of the polynomials and the input delay respectively. The optional parameter initC determines if initial
values for the Gauß-Newton algorithm should be used.

This algorithm calculates the derivatives of the cost function by an automatic differentiation package.

Use for example arxModelPem = PEM2(iddata,"arx",na,nb,0,na,nk);""" ->

function PEM2(iddata::iddataObject,method::ASCIIString,nff::Int64,nbb::Int64,ncc::Int64,ndd::Int64,nkk::Int64;initC::Bool=true)

  global nf = nff;   # we need to use global paramater in order to use this information within the ForwardDiff algorithms
  global nb = nbb;
  global nc = ncc;
  global nd = ndd;
  global nk = nkk;

  global y  = iddata.y;
  global u  = iddata.u;

  # specify modelfunction and time horizon for the model first
  if method == "arx"
    # specify initial X
    initialX = map(Float64,ones(nf+nb))

    # find time horizon (largest value going back in history)
    n1                   = nb+nk-1;
    n                    = findmax([n1; nf]);
    global timeHorizon   = n[1] + 1;

    # specify Model
    modelFunction        = arx_func;
  elseif method == "armax"
    # specify initial X
    if initC
      initCond   = init_cond(iddata.u, iddata.y, nf, nb, nc);
      initialX   = [initCond[1][2:nf+1] ; initCond[2][1:nb] ; initCond[3][2:nc+1]];
    else
      initialX   = map(Float64,ones(nf+nb+nc));
    end

    # find time horizon (largest value going back in history)
    n1                   = nb+nk-1;
    n                    = findmax([n1; nf ;nc]);
    global timeHorizon   = n[1] + 1;

    # specify Model
    modelFunction        = armax_func;
  elseif method == "oe"
    # specify initial X
    if initC
      initCond   = init_cond(iddata.u, iddata.y, nf, nb, nc);
      initialX   = [initCond[1][2:nf+1] ; initCond[2][1:nb] ];
    else
      initialX   = map(Float64,ones(nf+nb));
    end

    # find time horizon (largest value going back in history)
    n1                   = nb+nk-1;
    n                    = findmax([n1; nf]);
    global timeHorizon   = n[1] + 1;

    # specify Model
    modelFunction        = oe_func;
  else
    error("No matching method")
  end

  # Perform Gauß-Newton algorithm
  Model   = gaußnewton_forwarddiff(modelFunction,initialX,iddata.Ts,method);


  return Model;
end
