using PyPlot

#=   Function: calculateFit
Calculates a fit between real signal and predicted/simulated signal.

Author : Lars Lindemann @2015
                                                                    =#

function calculateFit(y::Array{Float64},y_pred::Array{Float64};method::ASCIIString="nrmse")
  # discard first samples, where filter has not settled
  if length(y)>200
    dismissSamples = 100;
    y1    = y[dismissSamples:end];
    y2    = y_pred[dismissSamples:end];
  else
    y1    = y;
    y2    = y_pred;
  end

  if length(y)!=length(y_pred)
    error("Real and predicted signal do not have the same size")
  end

  if method == "corcoeff"
    y_sigma     = sqrt(var(y1));
    y_predsigma = sqrt(var(y2));

    fit         = cov(y1,y2)[1]/(y_sigma*y_predsigma);
  elseif method == "nrmse"
    dif         = sqrt(sum((y1-y2).^2));
    norm        = sqrt(sum((y1-mean(y1)).^2));

    fit         = 1 - dif/norm;
  end

  return fit;
end

@doc """`fit = idModelOutput(iddata, idLinearModel,... [,method="simulation"])`

This function compares the real output y in iddata with the predicted or simulated output
of the inserted idLinearModels. The output is given by a vector that includes the fit.
Method can be choosen as "simulation" or "prediction"

Use for example fit = idModelOutput(iddata,arxModelPem,armaxModelPem,oeModelPem),
where the Models are PEM estimates.""" ->

function idModelOutput(iddata::iddataObject,kwargs...;method::ASCIIString="simulation")
  y = iddata.y;
  u = iddata.u;
  Ts= iddata.Ts;

  if length(y)!=length(u)
    error("Size of input and output must be the same")
  elseif Ts<=0
    error("Sampling time must be greater than zero")
  end

  # create a list for a legend
  legend_entry = ["Real data"];
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
    legend_entry = vcat(legend_entry,kwargs[i].method);
  end

  fit = zeros(Float64,length(kwargs));

  figure()
  t = Ts*collect(0:1:length(u)-1);
  plot(t,y)
  for i = 1:length(kwargs)
    # Check for stability of each model
    if !isstable(kwargs[i].G)
      error("Inserted model G of $((kwargs[i]).method) is unstable")
    end
    ##TODO substitute by reasonable(from prediction case)
    #=
    if !isstable(kwargs[i].H)
      error("Inserted model H of $((kwargs[i]).method) is unstable")
    end
    =#

    # Simulate system, plot it and calculate the fit
    if method == "simulation"
      y_pred, tt, x = lsim(kwargs[i].G,u,t);
    elseif method == "prediction"
      y_pred = calcOneStepAheadPrediction(iddata,kwargs[i].G,kwargs[i].H);
    else
      error("Method $(method) not available")
    end

    # Calculate fit of the prediction/simulation
    fit[i]       = calculateFit(y,y_pred);
    plot(t,y_pred)

  end
  legend(legend_entry)
  xlabel("Time")
  ylabel("Output")

  return fit;
end

@doc """`idFreqResp(idLinearModel,idSpectralModel,...])`

Plots the bode diagrams of idLinearModels or/and idSpectralModels.

Use for example idFreqResp(arxModelPem,armaxModelPem,spa1,etfe1),
where spa1 is a spa-function output, etfe1 a etfe-function ouput and the other parameters are PEM estimates.""" ->

function idFreqResp(kwargs...)
  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel) && !isa(kwargs[i],ControlKTH.idSpectralModel)
      error("Inserted parameter model $i is not of type idLinearModel/idSpectralModel")
    end
  end

  # create a list for a legend
  if length(kwargs)==1
    legend_entry = [kwargs[1].method];
  elseif length(kwargs)>1
    legend_entry = kwargs[1].method;
    for i = 2:length(kwargs)
      legend_entry = vcat(legend_entry,kwargs[i].method)
    end
  end

  figure()
  ax1 = subplot(211)
  ax1[:set_xscale]("log")
  ylabel("Magnitude(dB)")
  title("Bode diagram")
  ax2 = subplot(212)
  ax2[:set_xscale]("log")
  xlabel("Frequency (rad/s)")
  ylabel("Phase(deg)")

  # plot each model
  for i = 1:length(kwargs)
    if isa(kwargs[i],ControlKTH.idSpectralModel)
      # plotting for idSpectralModel
      ax1 = subplot(211)
      plot(kwargs[i].f*2*pi,20*log10(abs(kwargs[i].G)))
      legend(legend_entry)
      ax2 = subplot(212)
      ang = phaseunwrap(rad2deg(angle(kwargs[i].G)),false);
      plot(kwargs[i].f*2*pi,ang)
      legend(legend_entry)
    elseif isa(kwargs[i],ControlKTH.idLinearModel)
      # plotting for idLinearModel
      mag,ang,w = bode(kwargs[i].G);
      ax1 = subplot(211)
      plot(w,20*log10(squeeze(squeeze(mag,1),1)));
      legend(legend_entry)
      ax2 = subplot(212)
      ang = phaseunwrap(squeeze(squeeze(ang,1),1),false)
      plot(w,ang)
      legend(legend_entry)
    end
  end

end

@doc """`idAkaike(kwargs...])`

Calculates the akaike information criterion as in
aic = (1+2*d/N)*1/N*sum(y-y_pred)^2

kwargs need to contain idLinearModels.

Use for example idAkaike(arxModelPem,armaxModelPem),
where arxModelPem and armaxModelPem are PEM estimates.""" ->
# Linear Version
function idAkaike(kwargs...)
  aic = zeros(length(kwargs));

  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
  end

  for i = 1:length(kwargs)
    if (kwargs[i].method)[1:2]=="ar"
      # don't use nd here since the f and d polynomial are the same for arx/armax
      d      = kwargs[i].nf + kwargs[i].nb + kwargs[i].nc;
      aic[i] = (1+2*d/kwargs[i].N)*(kwargs[i].V/kwargs[i].N);
    else
      d      = kwargs[i].nf + kwargs[i].nb + kwargs[i].nc + kwargs[i].nd;
      aic[i] = (1+2*d/kwargs[i].N)*(kwargs[i].V/kwargs[i].N);
    end
  end

  return aic;
end




@doc """`idResid(iddata, idLinearModel,...])`

Plots the residuals of idLinearModels and iddata.

Use for example idResids(iddata, arxModelPem,armaxModelPem),
where arxModelPem and armaxModelPem are PEM estimates.""" ->
function idResid(iddata::iddataObject,kwargs...)
  # For more information see e.g. http://stattrek.com/estimation/confidence-interval.aspx
  # or see http://se.mathworks.com/help/signal/ug/confidence-intervals-for-sample-autocorrelation.html
  # Margin of error = critical value (1.96 for 95% interval) * standard error ( 1/sqrt(N))
  # Confidence interval = +-margin of error , since mean value is zero for white noise process
  maxLag    = 20;
  tauZero   = length(iddata.u);
  x         = -maxLag:maxLag;

  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
  end

  # calculate confidence interval for a white noise sequence
  ci_low   = -1.96/sqrt(length(iddata.u));
  ci_high  = 1.96/sqrt(length(iddata.u));

  # prepare plot and plot labels
  figure()
  ax1 = subplot(211)
  title("whiteness test - autocorrelation between residuals")
  ax2 = subplot(212)
  title("independence test - crosscorrelation between input and residuals")

  # create a list for a legend and plot it
  if length(kwargs)==1
    legend_entry = [kwargs[1].method];
  elseif length(kwargs)>1
    legend_entry = kwargs[1].method;
    for i = 2:length(kwargs)
      legend_entry = vcat(legend_entry,kwargs[i].method)
    end
  end


  for i = 1:length(kwargs)
    # calculate Resids of actual data
    y_pred    = calcOneStepAheadPrediction(iddata,kwargs[i].G,kwargs[i].H);
    res       = iddata.y-y_pred;

    # calculate auto- and crosscorrelation function
    var_res  = sum(res.^2);
    var_u    = sum((iddata.u).^2);
    R_resres = xcorr(res,res)/var_res;
    R_resu   = xcorr(res,iddata.u)/sqrt(var_res*var_u);


    # plot correlation functions
    ax1 = subplot(211)
    plot(x,R_resres[tauZero-maxLag:tauZero+maxLag])
    ax2 = subplot(212)
    plot(x,R_resu[tauZero-maxLag:tauZero+maxLag])
  end


  ax1 = subplot(211)
  axhline(ci_low,color="black",linestyle="--")
  axhline(ci_high,color="black",linestyle="--")
  legend(legend_entry)
  ax2 = subplot(212)
  axhline(ci_low,color="black",linestyle="--")
  axhline(ci_high,color="black",linestyle="--")
  legend(legend_entry)
end

@doc """`idPzPlot(idLinearModel,...])`

Plots the poles and zeros of idLinearModels.

Use for example idPzPlot(arxModelPem,armaxModelPem),
where arxModelPem and armaxModelPem are PEM estimates.""" ->

function idPzPlot(kwargs...)

  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
  end

  # prepare plot, plot labels and unit circle
  figure()
  title("Pole-Zero Plot")
  xlabel("Real Axis")
  ylabel("Imaginary Axis")
  x = collect(0:0.05:2*pi);
  plot(sin(x),cos(x),linestyle="--",color="black")
  if length(kwargs)<=7
    colormap = ["blue";"green";"red";"cyan";"magenta";"yellow";"black"];
  else
    error("Too many inserted models - maximum number is 7")
  end

  # create a list for a legend and plot it
  if length(kwargs)==1
    legend_entry = [kwargs[1].method];
  elseif length(kwargs)>1
    legend_entry = kwargs[1].method;
    for i = 2:length(kwargs)
      legend_entry = vcat(legend_entry,kwargs[i].method)
    end
  end

  # plot poles and zeros
  for ii = 1:length(kwargs)
    ny, nu = size(kwargs[ii].G, 1, 2)
    z, p, k = zpkdata(kwargs[ii].G)
    for i=1:ny, j=1:nu
      poles, zeros = p[i,j],z[i,j];
      # Plot poles and zeros
      plot(real(poles),imag(poles),"x",color=colormap[ii],label=legend_entry[ii])
      plot(real(zeros),imag(zeros),"o",color=colormap[ii])
    end
  end
  legend()
end

@doc """`idStep(idLinearModel,...])`

Plots the step response of idLinearModels.

Use for example idStep(arxModelPem,armaxModelPem),
where arxModelPem and armaxModelPem are PEM estimates.""" ->

function idStep(kwargs...)

  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
  end

  # prepare plot, plot labels and unit circle
  figure()
  title("Step Response")
  xlabel("Time in seconds")
  ylabel("Output of system")

  # create a list for a legend and plot it
  if length(kwargs)==1
    legend_entry = [kwargs[1].method];
  elseif length(kwargs)>1
    legend_entry = kwargs[1].method;
    for i = 2:length(kwargs)
      legend_entry = vcat(legend_entry,kwargs[i].method)
    end
  end

  # plot step response
  for i = 1:length(kwargs)
    if isstable(kwargs[i].G)
      y,t,x  = step(kwargs[i].G)
      plot(t,y,label=legend_entry[i])
    else
      warn("Inserted model $((kwargs[i]).method) is unstable")
    end
  end
  legend()
end

@doc """`idImpulse(idLinearModel,...])`

Plots the impulse response of idLinearModels.

Use for example idImpulse(arxModelPem,armaxModelPem),
where arxModelPem and armaxModelPem are PEM estimates.""" ->

function idImpulse(kwargs...)

  # Check validity of inputs
  for i = 1:length(kwargs)
    if !isa(kwargs[i],ControlKTH.idLinearModel)
      error("Inserted parameter model $i is not of type idLinearModel")
    end
  end

  # prepare plot, plot labels and unit circle
  figure()
  title("Impulse Response")
  xlabel("Time in seconds")
  ylabel("Output of system")

  # create a list for a legend and plot it
  if length(kwargs)==1
    legend_entry = [kwargs[1].method];
  elseif length(kwargs)>1
    legend_entry = kwargs[1].method;
    for i = 2:length(kwargs)
      legend_entry = vcat(legend_entry,kwargs[i].method)
    end
  end

  # plot step response
  for i = 1:length(kwargs)
    if isstable(kwargs[i].G)
      y,t,x  = impulse(kwargs[i].G)
      plot(t,y,label=legend_entry[i])
    else
      warn("Inserted model $((kwargs[i]).method) is unstable")
    end
  end
  legend()
end
