using ForwardDiff
using DSP

#=   Function: init_cond

Calculates inital start values for the Gauß-Newton Algorithm for
OE and Armax models

Author : Cristian Rojas
                                                                    =#

function init_cond(u, y, na, nb, nc)
    # INIT_COND Determines initial conditions for the parameters of an ARMAX model
    #
    # Input arguments:
    #  u:          Input data (column vector)
    #  y:          Output data (column vector)
    #  na, nb, nc: Degrees of polynomials A, B, C of the model to be estimated
    #
    # Output arguments:
    #  A,B,C:      Vector of initial conditions for the estimated parameters
    #

    trans = 50  # Number of initial data to be removed (due to transients)
    nk    = 20  # Orden of AR polynomial to be estimated

    # Determine data length
    N = length(u)

    # Apply the recursive least squares method
    theta  = zeros(na+nb,1)
    phi    = zeros(na+nb,1)
    P      = 10*var(y)*eye(na+nb)
    for i = 1:N
        e     = y[i] - phi'*theta
        temp  = 1 + phi'*P*phi
        P     = P - P*phi*phi'*P/temp[1]
        theta = theta + P*phi*e
        phi   = [-y[i]; phi[1:na-1]; u[i]; phi[na+1:na+nb-1]]
    end

    # Generate instruments
    z = zeros(na+nb,N+1)
    P = 10*var(y)*eye(na+nb)
    for i = 1:N
        z[:,i+1] = [-z[:,i]'*theta; z[1:na-1,i]; u[i]; z[na+1:na+nb-1,i]]
    end

    # Apply instrumental variables
    phi = zeros(na+nb,1)
    P   = 10*var(y)*eye(na+nb)
    #Th  = zeros(na+nb,0)
    for i = trans+1:N
        e     = y[i] - phi'*theta
        temp  = 1 + phi'*P*z[:,i+1]
        P     = P - P*z[:,i+1]*phi'*P/temp[1]
        theta = theta + P*z[:,i+1]*e
        phi   = [-y[i]; phi[1:na-1]; u[i]; phi[na+1:na+nb-1]]
    end

    # Obtain residuals v(t) = C*e(t)
    v   = zeros(N,1)
    phi = zeros(na+nb,1)
    for i = 1:N
        temp = phi'*theta
        v[i] = y[i] + temp[1]
        phi  = [y[i]; phi[1:na-1]; -u[i]; phi[na+1:na+nb-1]]
    end

    # Apply the recursive least squares method to estimate 1/C as an AR polynomial
    theta  = zeros(nk,1)
    phi    = zeros(nk,1)
    P      = 10*var(y)*eye(nk)
    for i = 1:N
        e     = v[i] - phi'*theta
        temp  = 1 + phi'*P*phi
        P     = P - P*phi*phi'*P/temp[1]
        theta = theta + P*phi*e
        phi   = [-v[i]; phi[1:nk-1]]
    end

    # Obtain e(t) = v(t) / C
    e   = zeros(N,1)
    phi = zeros(nk,1)
    for i = 1:N
        temp = phi'*theta
        e[i] = v[i] + temp[1]
        phi  = [-e[i]; phi[1:nk-1]]
    end

    # Estimate A, B, C from y, u and e
    if nc >= 1
        phi    = zeros(na+nb+nc,1)
        theta  = zeros(na+nb+nc,1)
        P      = 10*var(y)*eye(na+nb+nc)

        for i = trans+1:N
            temp  = phi'*theta
            r     = y[i] - temp[1]
            temp  = 1 + phi'*P*phi
            P     = P - P*phi*phi'*P/temp[1]
            theta = theta + P*phi*r
            phi   = [-y[i]; phi[1:na-1]; u[i]; phi[na+1:na+nb-1]; e[i]; phi[na+nb+1:na+nb+nc-1]]
        end

        A = [1; theta[1:na]]
        B = theta[na+1:na+nb]
        C = [1; theta[na+nb+1:na+nb+nc]]

    else
        phi    = zeros(na+nb,1)
        theta  = zeros(na+nb,1)
        P      = 10*var(y)*eye(na+nb)
        for i = trans+1:N
            temp  = phi'*theta
            r     = y[i] - temp[1]
            temp  = 1 + phi'*P*phi
            P     = P - P*phi*phi'*P/temp[1]
            theta = theta + P*phi*r
            phi   = [-y[i]; phi[1:na-1]; u[i]; phi[na+1:na+nb-1]]
        end

        A = [1; theta[1:na]]
        B = theta[na+1:na+nb]
        C = [1]
    end

    return((A, B, C))
end

#=   Function: gaußnewton_forwarddiff

Performs the Gauß-Newton Algorithm by using the package ForwardDiff,
which performs automatic differentiation.

Author : Lars Lindemann @2015
                                                                    =#

function gaußnewton_forwarddiff(modelFunction::Function,initialTheta::Array{Float64},Ts::Float64,
                                method::ASCIIString;stabilityFix::Bool=false,stepSizeControl::Bool=true)

  maxIterations= 30;
  exitTreshold = 0.0001;
  theta        = initialTheta;
  T            = theta;
  V            = zeros(maxIterations);
  mu           = 1;
  V_out        = 0;
  N            = length(y)-timeHorizon+1;

  result = false;
  counter = 1;

  while(!result)

    # calculate gradient and hession of the model function
    j = forwarddiff_gradient(modelFunction, Float64, fadtype=:typed)
    g = forwarddiff_hessian(modelFunction, Float64, fadtype=:typed)

    V_g        = j(theta);
    V_h        = g(theta);
    V[counter] = (VV.d).v; # see special format of hessian type to understand this notation

    if counter == 1
      theta = theta - mu*(V_h\V_g);
      T     = [T theta];
    # if MSE got worse in the last step, control the step size
    elseif ( V[counter]>V[counter-1] || isnan(V[counter]) ) && stepSizeControl
      # reset counter and lower step size
      counter    -= 1;
      mu         *= 0.1;
      # calculate new theta
      theta = T[:,size(T,2)-1];
      V_g          = j(theta);
      V_h          = g(theta);
      V[counter+1] = (VV.d).v;
      theta = theta - mu*(V_h\V_g);
      T[:,size(T,2)] = theta;
    # if MSE got better in the last step
    else
      # reset mu to 1
      mu = 1;

      # exit criterion after a successfull step
      if abs(V[counter]-V[counter-1])<exitTreshold
        result = true;
        V_out  = V[counter]*N;
      elseif counter == maxIterations
        result = true
        V_out  = V[counter]*N;
      else
        # use steepest descent
        theta = theta - mu*(V_h\V_g);
        T     = [T theta];
      end

    end

    # check for stability and inverse roots if necessary
    if method=="armax" && stabilityFix
      theta[nf+nb+1:nf+nb+nc]   = checkStability(theta[nf+nb+1:nf+nb+nc]);
      theta[1:nf]               = checkStability(theta[1:nf]);
    elseif method=="oe" && stabilityFix
      theta[1:nf]   = checkStability(theta[1:nf]);
    end

    counter +=1;
  end

  # build idModel as output
  if method == "arx"
    Model = createModelOutput(theta,"arx",nf,nb,0,nf,nk,Ts,V_out,N)
  elseif method == "armax"
    Model = createModelOutput(theta,"armax",nf,nb,nc,nf,nk,Ts,V_out,N)
  elseif method == "oe"
    Model = createModelOutput(theta,"oe",nf,nb,0,0,nk,Ts,V_out,N)
  end

  return Model;
end

#=   Function: preprocessing

Gives several preprocessing options and uses the DSP toolkit

Author : Lars Lindemann @2015
                                                                    =#
@doc """`iddata = preprocessing(iddata, method, [cutoff=0.25, Ts=1.0, order=10])`

Performs preprocessing for iddata with the options "lowpass", "highpass" and "mean" that should be given as a string for method. If Ts within
the optional arguments isn't stated, cutoff is assumed to be the normalized frequency. Butterworth filters are used. """ ->
function preprocessing(iddata::iddataObject,method::ASCIIString;cutoff::Float64=0.25,Ts::Float64=1.,order::Int64=10)

  if method == "lowpass"
    lowpass        = digitalfilter(Lowpass(cutoff,fs=1/Ts),Butterworth(order));
    iddata.u       = filt(lowpass,iddata.u);
    iddata.y       = filt(lowpass,iddata.y);
  elseif method == "highpass"
    highpass       = digitalfilter(Highpass(cutoff,fs=1/Ts),Butterworth(order));
    iddata.u       = filt(highpass,iddata.u);
    iddata.y       = filt(highpass,iddata.y);
  elseif method == "mean"
    iddata.u       -= mean(iddata.u);
    iddata.y       -= mean(iddata.y);
  end

  return iddata
end

#=  Function: normalizeCorr

Normalization of auto-and crosscorrelation
The correlation should be estimated as:
R_u(tau) = 1/N sum_k^N u(k)*u(k-tau)
For a given time series, the xcorr function returns the not normalized
cross/-autocorrelation (without 1/N). Since different lags result in a
different amount of samples that can be used for the estimation (finite time series),
it is useful to normalize. This is done in an unbiased way so far
R_u,unbiased(tau) = 1/(N-abs(tau)) R_u(tau)
or a biased way
R_u,biased(tau)   = 1/N R_u(tau)

compare Matlab commands
xcorr(u,u,'unbiased') or xcorr(u,u,'biased')

Author : Lars Lindemann @2015
                                                                    =#
function normalizeCorr(R_u::Array{Float64,1};method::ASCIIString="unbiased",option="two-sided")
  # correlation function should have the length 2*N-1 in every case! calculate the zero lag position
  # tauZero is equal to N
  tauZero = round(Integer,(length(R_u)+1)/2);
  N       = tauZero;

  if method=="unbiased"
    # unbiased normalization
    for lag=0:tauZero-1
      R_u[tauZero+lag] = 1/(N-lag)*R_u[tauZero+lag];
      R_u[tauZero-lag] = 1/(N-lag)*R_u[tauZero-lag];
    end
  elseif method=="biased"
    R_u   = R_u/N
  end

  # two return options
  if option=="two-sided"
    return R_u
  elseif option=="one-sided"
    return R_u[tauZero:end]
  end
end


#=   Function: selectWindow

Simply gives a window with the choosen method and length as output

Author : Lars Lindemann @2015
                                                                    =#

function selectWindow(window::ASCIIString,N::Int64)

  if window=="rect"
    win = rect(N);
  elseif window=="hanning"
    win = hanning(N);
  elseif window=="hamming"
    win = hamming(N);
  elseif window=="tukey"
    win = tukey(N);
  elseif window=="blackman"
    win = blackman(N);
  else
    error("Window not available")
  end

  return win
end

#=   Function: checkStability

Checks stability for a given polynomial A. In this special case the functions
takes the polynomial without the '1' in the beginning, which is characteristic
for F, C and D polynomial for PEM models.
In case of an unstable pole, this pole is inversed.

Author : Lars Lindemann @2015
                                                                    =#

function checkStability(A::Array{Float64})

  if isnan(A[1])
    error("Algorithm failed to converge")
  end

  rootPoly = Poly([1 ; A]);
  rootsA   = roots(rootPoly);
  maxRoot  = findmax(abs(rootsA));
  if maxRoot[1]>1
    for i=1:length(rootsA)
      if abs(rootsA[i])>1
        rootsA[i] = 1/rootsA[i];
      end
    end
    A  = (poly(rootsA).a)[2:end];
  end

  return A;
end

#=   Function: createModelOutput

Creates an idLinearModel for PEM and the methods arx, armax, oe and bj.

Author : Lars Lindemann @2015
                                                                    =#
function createModelOutput(theta::Array{Float64},method::ASCIIString,nf::Int64,nb::Int64,nc::Int64,nd::Int64,nk::Int64,
                           Ts::Float64,V::Float64=0.,N::Int64=0)

  if method == "armax"
    if (nk+nb-1)>=nf || (nc>=nf)
      if (nk+nb-1)>=nc
        numG   = theta[nf+1:nf+nb];
        numH   = [1;theta[nf+nb+1:nf+nb+nc];zeros(-nc+nk+nb-1)];
        den    = [1;theta[1:nf];zeros(-nf+nk+nb-1)];
      elseif (nk+nb-1)<nc
        numG   = [theta[nf+1:nf+nb];zeros(-nk-nb+1+nc)];
        numH   = [1;theta[nf+nb+1:nf+nb+nc]];
        den    = [1;theta[1:nf];zeros(-nf+nc)];
      end
    else
      numG   = [theta[nf+1:nf+nb];zeros(-nk-nb+1+nf)];
      numH   = [1;theta[nf+nb+1:nf+nb+nc];zeros(-nc+nf)];
      den    = [1;theta[1:nf]];
    end

    G   =   tf(numG,den,Ts);
    H   =   tf(numH,den,Ts);
    Model = idLinearModel(G,H,nb,nc,nf,nf,nk,"armax$nf$nb$nc$nk",V,N);

  elseif method == "oe"
    if (nk+nb-1)>nf
      num   = collect(theta[nf+1:nf+nb]);
      den   = [1;theta[1:nf];zeros(nk+nb-1-nf)];
    elseif (nk+nb-1)<=nf
      num   = [theta[nf+1:nf+nb];zeros(nf-nk-nb+1)];
      den   = [1;theta[1:nf]];
    end

    G   =   tf(num,den,Ts);
    H   =   tf([1],[1],Ts);
    Model = idLinearModel(G,H,nb,0,0,nf,nk,"oe$nb$nf$nk",V,N);

  elseif method == "arx"
    if (nk+nb-1)>nf
      num   = theta[nf+1:nf+nb];
      den   = [1;theta[1:nf];zeros(nk+nb-1-nf)];
    elseif (nk+nb-1)<=nf
      num   = [theta[nf+1:nf+nb];zeros(nf-nk-nb+1)];
      den   = [1;theta[1:nf]];
    end

    G   =   tf(num,den,Ts);
    H   =   tf([1;zeros(length(den)-1)],den,Ts);
    Model = idLinearModel(G,H,nb,0,nf,nf,nk,"arx$nf$nb$nk",V,N);

  elseif method == "bj"
    if (nk+nb-1)>nf
      numG   = theta[nf+1:nf+nb];
      denG   = [1;theta[1:nf];zeros(nk+nb-1-nf)];
    elseif (nk+nb-1)<=nf
      numG   = [theta[nf+1:nf+nb];zeros(nf-nk-nb+1)];
      denG   = [1;theta[1:nf]];
    end

    if nc>nd
      numH   = [1;theta[nf+nb+1:nf+nb+nc]];
      denH   = [1;theta[nf+nb+nc+1:nf+nb+nc+nd];zeros(nc-nd)];
    elseif nc<=nd
      numH   = [1;theta[nf+nb+1:nf+nb+nc];zeros(nd-nc)];
      denH   = [1;theta[nf+nb+nc+1:nf+nb+nc+nd]];
    end
    G   =   tf(numG,denG,Ts);
    H   =   tf(numH,denH,Ts);
    Model = idLinearModel(G,H,nb,nc,nd,nf,nk,"bj$nf$nb$nc$nd$nk",V,N);

  end

  return Model
end

#=   Function: calcOneStepAheadPrediction

Calculates the one step ahead prediction as in
y_pred(n) = [1-H^-1]*y(n) + H^-1*G*u(n)

Author : Lars Lindemann @2015
                                                                    =#
function calcOneStepAheadPrediction(iddata::iddataObject,G::ControlKTH.LTISystem,H::ControlKTH.LTISystem)
  t         = iddata.Ts*collect(0:1:length(iddata.u)-1);
  G_1       = 1-1/H;
  G_2       = 1/H * G;
  y_1, t    = lsim(G_1,iddata.y,t);
  y_2, t    = lsim(G_2,iddata.u,t);

  if issiso(G)
    ndims(y_1)>1 ? y_1=squeeze(y_1,2) : y_1;
    ndims(y_2)>1 ? y_2=squeeze(y_2,2) : y_2;
  end
  y_pred    = y_1 + y_2;

  return y_pred;
end
