# Functions for calculating time response of a system

# XXX : `step` is a function in Base, with a different meaning than it has
# here. This shouldn't be an issue, but it might be.
@doc """`[y, t, x] = step(sys[, Tf])` or `[y, t, x] = step(sys[, t])`

Calculate the step response of system `sys`. If the final time `Tf` or time
vector `t` is not provided, one is calculated based on the system pole
locations.""" ->
function Base.step(sys::StateSpace, t::AbstractVector)
    lt = length(t)
    ny, nu = size(sys)
    nx = sys.nx
    u = ones(size(t))
    x0 = zeros(nx, 1)
    if nu == 1
        y, t, x = lsim(sys, u, t, x0, :zoh)
    else
        x = Array(Float64, lt, nx, nu)
        y = Array(Float64, lt, ny, nu)
        for i=1:nu
            y[:,:,i], t, x[:,:,i] = lsim(sys[:,i], u, t, x0, :zoh)
        end
    end
    return y, t, x
end
Base.step(sys::StateSpace, Tf::Real) = step(sys, _default_time_vector(sys, Tf))
Base.step(sys::StateSpace) = step(sys, _default_time_vector(sys))
Base.step(sys::LTISystem, args...) = step(ss(sys), args...)


@doc """`[y, t, x] = impulse(sys[, Tf])` or `[y, t, x] = impulse(sys[, t])`

Calculate the impulse response of system `sys`. If the final time `Tf` or time
vector `t` is not provided, one is calculated based on the system pole
locations.""" ->
function impulse(sys::StateSpace, t::AbstractVector)
    lt = length(t)
    ny, nu = size(sys)
    nx = sys.nx
    u = zeros(size(t))
    if iscontinuous(sys)
        # impulse response equivalent to unforced response of
        # ss(A, 0, C, 0) with x0 = B.
        imp_sys = ss(sys.A, zeros(nx, 1), sys.C, zeros(ny, 1))
        x0s = sys.B
    else
        imp_sys = sys
        x0s = zeros(nx, nu)
        u[1] = 1/sys.Ts
    end
    if nu == 1
        y, t, x = lsim(sys, u, t, x0s, :zoh)
    else
        x = Array(Float64, lt, nx, nu)
        y = Array(Float64, lt, ny, nu)
        for i=1:nu
            y[:,:,i], t, x[:,:,i] = lsim(sys[:,i], u, t, x0s[:,i], :zoh)
        end
    end
    return y, t, x
end
impulse(sys::StateSpace, Tf::Real) = impulse(sys, _default_time_vector(sys, Tf))
impulse(sys::StateSpace) = impulse(sys, _default_time_vector(sys))
impulse(sys::LTISystem, args...) = impulse(ss(sys), args...)


@doc """`[y, t, x] = lsim(sys, u, t[, x0, method])`

Calculate the time response of system `sys` to input `u`. If `x0` is ommitted,
a zero vector is used.

Continuous time systems are discretized before simulation. By default, the
method is chosen based on the smoothness of the input signal. Optionally, the
`method` parameter can be specified as either `:zoh` or `:foh`.""" ->
function lsim(sys::StateSpace, u::AbstractVecOrMat, t::AbstractVector,
        x0::VecOrMat=zeros(sys.nx, 1), method::Symbol=_issmooth(u) ? :foh : :zoh)
    ny, nu = size(sys)
    nx = sys.nx

    if length(x0) != nx
        error("size(x0) must match the number of states of sys")
    elseif !any(size(u) .== [(length(t), nu) (length(t),)])
        error("u must be of size (length(t), nu)")
    end

    dt = Float64(t[2] - t[1])
    if !iscontinuous(sys) || method == :zoh
        if iscontinuous(sys)
            dsys = c2d(sys, dt, :zoh)[1]
        else
            if sys.Ts != dt
                error("Time vector must match sample time for discrete system")
            end
            dsys = sys
        end
    else
        dsys, x0map = c2d(sys, dt, :foh)
        x0 = x0map*[x0; u[1,:].']
    end
    x = ltitr(dsys.A, dsys.B, map(Float64,u), map(Float64,x0))
    y = (sys.C*(x.') + sys.D*(u.')).'
    return y, t, x
end
lsim(sys::LTISystem, args...) = lsim(ss(sys), args...)


@doc """`ltitr(A, B, u[, x0])`

Simulate the discrete time system `x[k + 1] = A x[k] + B u[k]`, returning `x`.
If `x0` is not provided, a zero-vector is used.""" ->
function ltitr{T}(A::Matrix{T}, B::Matrix{T}, u::AbstractVecOrMat{T},
        x0::VecOrMat{T})
    n = size(u, 1)
    x = Array(T, size(A, 1), n)
    for i=1:n
        x[:,i] = x0
        x0 = A * x0 + B * u[i,:].'
    end
    return x.'
end
ltitr{T}(A::Matrix{T}, B::Matrix{T}, u::AbstractVecOrMat{T}) =
        ltitr(A, B, u, zeros(T, size(A, 1), 1))

# STEPINFO FUNCTIONS

# Helper type:
typealias NumOrArray{T<:Real} Union(T, Array{T})
typealias NumArray{T<:Real} Array{T}
typealias RangeOrArray{T<:Real} Union{NumArray{T}, Range{T}}

# Warn in case of mimo
sisocheck(sys::LTISystem, funcname::ASCIIString) = !issiso(sys) && error("$funcname only defined for siso systems.")

@doc """`M = overshoot(sys, t=...)`
Calculate the percentage overshoot of system `sys` relative to its final value. The time vector used for step response calculations can be provided manually.

`M = overshoot(y, yfinal)`

Calculate the percentage overshoot of step response `y` with final value `yfinal`. If no final value is provided it is approximated using `yfinal = y[end]`.
""" ->
function overshoot(sys::LTISystem; t::AbstractVector=[NaN])
    sisocheck(sys, "Overshoot")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)
    return overshoot(vec(y), finalvalue(sys))
end
function overshoot(y::AbstractVector, yf::Real=y[end])
    return isfinite(yf) ? 100*max(0, peak(y)-yf)/abs(yf) : NaN
end


@doc """`M = undershoot(sys, t=...)`
Calculate the percentage undershoot of system `sys` relative to its final value. The time vector used for step response calculations can be provided manually.

`M = undershoot(y, yfinal)`

Calculate the percentage undershoot of step response `y` with final value `yfinal`. If no final value is provided it is approximated using `yfinal = y[end]`.
""" ->
function undershoot(sys::LTISystem; t::AbstractVector=[NaN])
    sisocheck(sys, "Overshoot")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)
    return undershoot(vec(y), finalvalue(sys))
end
function undershoot(y::AbstractVector, yf::Real=y[end])
    return isfinite(yf) ? -100*min(0, peak(y)-yf)/abs(yf) : NaN
end


@doc """`yfinal = finalvalue(sys)`
Calculate the final value of the step response of system `sys`""" ->
function finalvalue(sys::LTISystem)
    !issiso(sys) && error("Final value only defined for siso systems.")
    ny, nu = size(sys)
    # if sys is stable use dcgain, if not determine if the system diverges to plus or minus Inf.
    if iscontinuous(sys)
        return isstable(sys) ? dcgain(sys)[1] : sign(gain(sys))*Inf
    else
        return isstable(sys) ? evalfr(sys, 1)[1] : sign(gain(sys))*Inf
    end
end
finalvalue(y::AbstractVector) = y[end]
finalvalue(y::Real) = y


@doc """`ypeak = peak(sys, t=...)`
Calculate the peak value of the step response of system `sys`. The time vector used for step response calculations can be provided manually.""" ->
function peak(sys::LTISystem; t::AbstractVector=[NaN])
    sisocheck(sys, "Peak")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)
    return peak(vec(y), finalvalue(sys))
end
peak(y::AbstractVector, yf::Real=y[end]) = isfinite(yf) ? sort(y)[end] : Inf
peak(y::Real) = y

@doc """`tpeak = peaktime(sys, t=...)`
Calculate the peak time of the step response of system `sys`. The time vector used for step response calculations can be provided manually.""" ->
function peaktime(sys::LTISystem; t::AbstractVector=[NaN])
    sisocheck(sys, "Peak time")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)

    return peaktime(vec(y), t, finalvalue(sys))
end
function peaktime(
    y::AbstractVector,
    t::AbstractVector = 1.:length(y),
    yf::Real = y[end]
    )

    length(y) != length(t) && error("y and t must have the same length")
    return !isfinite(yf) && yf > 0 ? Inf : t[indmax(y)]
end

 @doc """`trise = risetime(sys, rtlimits=..., t=...)`

 Calculates the rise time of the step response of system `sys`.

 `trise = risetime(y, rtlimits=..., t=..., yf=..., Ts=...)`

 Calculates the rise time of step response `y` with time vector `t` and final value `yf`. If no time vector or final value is provided `Ts = 1` and `yf = y[end]` is used.

  The rise time of a step response is the time it takes to rise from `rtlimits[1]`  to `rtlimits[2]` (rise time limits) of its steady state value. If no limits are provided `rtlimits = [0.1, 0.9]` is used.""" ->
 function risetime(sys::LTISystem; rtlimits::AbstractVector=[.1, .9], t::AbstractVector=[NaN])
    sisocheck(sys, "Rise time")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)

    return risetime(vec(y), collect(t), rtlimits, finalvalue(sys), sys.Ts)
end
function risetime(
    y::AbstractVector,
    t::AbstractVector=collect(1.:length(y)),
    rtlimits::AbstractVector=[.1, .9],
    yf::Real=y[end],
    Ts::Real=0.
    )
    # Calculate rise time for SISO step response

    # Check inputs
    (rtlimits[1]<=0. || rtlimits[2]>=1. || rtlimits[1]>rtlimits[2]) && error("Invalid rise time limits")
    length(y) <= 1 && error("Step response must contain more than one sample")
    length(y) != length(t) && error("y and t must have the same length")
    !isfinite(yf) && return NaN


    # Rise time is undefined if yf < y0
    y0 = y[1]
    y0 >= yf && return NaN

    # Find the first index for which y has passes the lower limit defined by rtlimits[1]
    ylow = y0 + rtlimits[1]*(yf - y0)
    ilow = findnext(x -> x >= ylow, y, 1)


    if ilow == 0
        # rlimits[1] has not been reached
        return NaN
    elseif y[ilow] != y[ilow-1] && Ts == 0
        # Iterpolate for accuracy
        tlow = t[ilow] + (t[ilow]-t[ilow-1]) * (ylow-y[ilow]) / (y[ilow]-y[ilow-1])
    else
        tlow = t[ilow]
    end

    # Find the first index for which y has passes the upper limit defined by rtlimits[2]
    yhigh = y0 + rtlimits[2]*(yf - y0)
    ihigh = findnext(x -> x >= yhigh, y, 1)

    if ihigh == 0
        # rlimits[2] has not been reached
        return NaN
    elseif y[ihigh] != y[ihigh-1] && Ts == 0
        thigh = t[ihigh] + (t[ihigh]-t[ihigh-1]) * (yhigh-y[ihigh]) / (y[ihigh]-y[ihigh-1])
    else
        thigh = t[ihigh]
    end

    return thigh - tlow
end
function risetime(
    y::AbstractVector;
    t::AbstractVector=collect(1.:length(y)),
    rtlimits::AbstractVector=[.1, .9],
    yf::Real=y[end],
    Ts::Real=0.
    )
    return risetime(y, t, rtlimits, yf, Ts)
end

 @doc """`tset = settingtime(sys, threshold=..., t=...)`

 Calculates the setting time of the step response of system `sys`.

 `tset = settingtime(y, threshold=..., t=..., yf=..., Ts=...)`

 Calculates the setting time of step response `y` with time vector `t` and final value `yf`. If no time vector or final value is provided `Ts = 1` and `yf = y[end]` is used.

  The setting time of a step response is the time at which the error `|y-yf|` becomes smaller than a fraction `threshold` of its maximum value. If no threshold is provided `threshold = 0.02` (2%) is used.""" ->
function settingtime(sys::LTISystem; threshold::Real=0.02, t::AbstractVector=[NaN])
    sisocheck(sys, "Setting time")
    y,t = isnan(t[1]) ? step(sys) : step(sys, t)

    return settingtime(vec(y), collect(t), threshold, finalvalue(sys), sys.Ts)
end
function settingtime(
    y::AbstractVector,
    t::AbstractVector=collect(1.:length(y)),
    threshold::Real=0.02,
    yf::Real=y[end],
    Ts::Real=0.
    )
    # Check inputs
    0 < threshold < 1 || error("Invalid setting time threshold")
    length(y) != length(t) && error("y and t must have the same length")
    !isfinite(yf) && return NaN

    err = abs(y-yf)
    tol = threshold*sort(err)[end]
    #Find the last index for which the threshold is not passed
    i = find(x->x > tol, err)
    isempty(i) ? (return 0) : iSettle = i[end]

    if iSettle == length(y)
        # Step response hasn't settled
        return NaN
    elseif y[iSettle] != y[iSettle+1] && Ts == 0
        # Interpolate for more accuracy if the system is continous
        ySettle = yf + sign(y[iSettle]-yf)*tol
        return t[iSettle] + (t[iSettle]-t[iSettle+1])/(y[iSettle]-y[iSettle+1]) * (ySettle-y[iSettle])
    else
        return t[iSettle+1]
    end
end
function settingtime(
    y::AbstractVector;
    t::AbstractVector=collect(1.:length(y[:,1,1])),
    threshold::Real=0.02,
    yf::Real=y[end],
    Ts::Real=0.
    )
    settingtime(y, t, threshold, yf, Ts)
end

type StepInfo
    RiseTime::Float64
    SettingTime::Float64
    Overshoot::Float64
    Undershoot::Float64
    Peak::Float64
    PeakTime::Float64
    FinalValue::Float64
    function StepInfo(sys::LTISystem, t::AbstractVector=[NaN])
        !issiso(sys) && error("stepinfo only accepts siso systems")
        y,t = isnan(t[1]) ? step(sys) : step(sys, t)
        yvec = vec(y)
        tvec = collect(t)
        yf = finalvalue(sys)
        risetime_ = risetime(yvec, tvec, [.1,.9], yf, sys.Ts)
        settingtime_ = settingtime(yvec, tvec, 0.02, yf, sys.Ts)
        overshoot_ = overshoot(yvec, yf)
        undershoot_ = undershoot(yvec, yf)
        peak_ = peak(yvec, yf)
        peaktime_ = peaktime(yvec, tvec, yf)
        finalvalue_ = yf
        return new(risetime_, settingtime_, overshoot_, undershoot_, peak_, peaktime_, finalvalue_)
    end
end
stepinfo(sys::LTISystem; t::AbstractVector=[NaN]) = StepInfo(sys, t)
function Base.show(io::IO, sinfo::StepInfo)
    println(io,"RiseTime:    $(sinfo.RiseTime)")
    println(io,"SettingTime: $(sinfo.SettingTime)")
    println(io,"Overshoot:   $(sinfo.Overshoot)")
    println(io,"Undershoot:  $(sinfo.Undershoot)")
    println(io,"Peak:        $(sinfo.Peak)")
    println(io,"PeakTime:    $(sinfo.PeakTime)")
    println(io,"FinalValue:  $(sinfo.FinalValue)")
end

# HELPERS:

# TODO: This is a poor heuristic to estimate a "good" time vector to use for
# simulation, in cases when one isn't provided.
function _default_time_vector(sys::LTISystem, Tf::Real=-1)
    Ts = _default_Ts(sys)
    if Tf == -1
        Tf = _default_Tf(sys, Ts)
    end
    return 0:Ts:Tf
end


function _default_Ts(sys::LTISystem)
    if !iscontinuous(sys)
        Ts = sys.Ts
    else
        ps = pole(sys)
        if !isstable(ps)
            Ts = 0.05
        else
            r = minimum(abs(ps))
            Ts = 0.07/(r != 0 ? r : 1)
        end
    end
    return Ts
end

function _default_Tf(sys::LTISystem, Ts::Float64)
    # calculate a final time for each siso system and pick the largest one
    ny, nu = size(sys)
    final_times = Array(Float64, ny, nu)
    for i=1:ny, j=1:nu
        pstmp = pole(sys[i,j])
        if !isstable(pstmp)
            final_times[i,j] = 100*Ts
        else
            ps = pstmp[(imag(pstmp) .>= 0) & (abs(real(pstmp)) .> 0)]
            final_times[i,j] = norm( 6.9./abs(real(ps)) )
        end
    end
    return maximum(final_times)
end

# Determine if a signal is "smooth"
function _issmooth(u::Array, thresh::FloatingPoint=0.75)
    u = [zeros(1, size(u, 2)); u]       # Start from 0 signal always
    range = maximum(u) - minimum(u)
    du = abs(diff(u))
    return !isempty(du) && all(maximum(du) <= thresh*range)
end

# # Old _default_time_vector
# function _default_time_vector(sys::LTISystem, Tf::Real=-1)
#     Ts = _default_Ts(sys)
#     if Tf == -1
#         Tf = 100*Ts
#     end
#     return 0:Ts:Tf
# end

# # Old _default_Ts
# if !iscontinuous(sys)
#     Ts = sys.Ts
# elseif !isstable(sys)
#     Ts = 0.05
# else
#     ps = pole(sys)
#     r = minimum(abs(ps))
#     if r == 0.0
#         r = 1.0
#     end
#     Ts = 0.07/r
# end
# return Ts
