##### ss2tf-function #####
#=
   num,den = ss2tf(A,B,C,D,iu)  calculates the transfer function:

               NUM(s)          -1
       H(s) = -------- = C(sI-A) B + D
               DEN(s)
   of the system:
       .
       x = Ax + Bu
       y = Cx + Du

   from the iu'th input.  Vector DEN contains the coefficients of the
   denominator in descending powers of s.  The numerator coefficients
   are returned in matrix NUM with as many rows as there are
   outputs y.
=#

function ss2tf(A::Array,B::Array,C::Array,D::Array,ni::Int64=1)

  if ni>size(B,2)
    error("ni is not allowed to be bigger than the number of inputs");
  end

  # Create state space model from function inputs
  sys = ss(A,B[:,ni],C,D[:,ni]);

  # Calculate zeros, poles and gains for simpler conversion to tf
  z,p,k   = zpkdata(sys);
  p       = p[1];

  # build denominator polynomial
  den = poly(p);

  # build numerator polynomial
  ny,nu = size(k);

  if nu==0
    num =zeros(ny,0);
  else
    if ndims(den.a)==1
      lden = length(den.a);
    else
      dim1,dim2 = size(den.a);
      lden = dim1 * dim2;
    end
    num = zeros(ny,lden);
    for j=1:ny
      zj = z[j]; #since all elements given by the zpkdata function are finite!
      pzj = poly(zj);
      lzj = length(zj);
      num[j,lden-lzj:lden] = k[j] * pzj.a;
    end
  end

  # now build up the whole transfer function
  return (num,den.a')
end


function Base.convert(::Type{TransferFunction}, sys::StateSpace)
    ny, nu = size(sys)
    z, p, k = zpkdata(sys)
    matrix = Array(SisoTf, ny, nu)
    for i=1:ny, j=1:nu
        # Calculate the num and den polynomials from the roots of the LTISystem. Use real() to make sure all coefficients are real.
        matrix[i,j] = SisoTf(real(poly(z[i,j]))*k[i,j], real(poly(p[i,j])))
    end
    return TransferFunction(matrix, sys.Ts, sys.inputnames, sys.outputnames)
end
