function Base.convert(::Type{StateSpace}, t::Union{TransferFunction, ZPK})
    if !isproper(t)
        error("System is improper, a state-space representation is impossible")
    end
    ny, nu = size(t)
    if isa(t, TransferFunction)
        mat = t.matrix
    else
        # input has type ZPK
        mat = [t.k[i,j]*SisoTf(real(poly(t.z[i,j])), real(poly(t.p[i,j]))) for i=1:ny, j=1:nu]
    end

    # TODO : These are added due to scoped for blocks, but is a hack. This
    # could be much cleaner.
    Ac = Bc = Cc = Dc = A = B = C = D = Array(Float64, 0, 0)
    for i=1:nu
        for j=1:ny
            a, b, c, d = siso_tf_to_ss(mat[j, i])
            if j > 1
                # vcat
                Ac = blkdiag(Ac, a)
                Bc = vcat(Bc, b)
                Cc = blkdiag(Cc, c)
                Dc = vcat(Dc, d)
            else
                Ac, Bc, Cc, Dc = (a, b, c, d)
            end
        end
        if i > 1
            # hcat
            A = blkdiag(A, Ac)
            B = blkdiag(B, Bc)
            C = hcat(C, Cc)
            D = hcat(D, Dc)
        else
            A, B, C, D = (Ac, Bc, Cc, Dc)
        end
    end
    A, B, C = balance_statespace(A, B, C)[1:3]
    return ss(A, B, C, D, t.Ts, inputnames=t.inputnames, outputnames=t.outputnames)
end

Base.promote_rule(::Type{StateSpace}, ::Type{TransferFunction}) = StateSpace

function siso_tf_to_ss(t::SisoTf)
    t = normalize_tf(t)
    tnum = num(t)
    tden = den(t)
    len = length(tden)
    d = Array(Float64, 1, 1)
    d[1] = tnum[1]

    if len==1 || tnum == zero(Poly{Float64})
        a = zeros(0, 0)
        b = zeros(0, 1)
        c = zeros(1, 0)
    else
        tden = tden[2:end]
        a = [-tden' ; eye(len - 2, len - 1)]
        b = eye(len - 1, 1)
        c = tnum[2:len]' - d * tden[:]'
    end
    return float64mat(a), float64mat(b), float64mat(c), d
end

function normalize_tf(t::SisoTf)
    d = t.den[1]
    return SisoTf(t.num/d, t.den/d)
end

function balance_statespace{S}(A::Matrix{S}, B::Matrix{S},
        C::Matrix{S}, perm::Bool=false)
    nx = size(A, 1)
    nu = size(B,2)
    ny = size(C,1)

    # Compute the transformation matrix
    mag_A = abs(A)
    mag_B = maximum([abs(B)  zeros(S, nx, 1)], 2)
    mag_C = maximum([abs(C); zeros(S, 1, nx)], 1)
    T = balance_transform(mag_A, mag_B, mag_C, perm)

    # Perform the transformation
    A = T*A/T
    B = T*B
    C = C/T

    return A, B, C, T
end

# Computes a balancing transformation `T` that attempts to scale the system so
# that the row and column norms of [T*A/T T*B; C/T 0] are approximately equal.
# If `perm=true`, the states in `A` are allowed to be reordered.
function balance_transform{R}(A::Matrix{R}, B::Matrix{R}, C::Matrix{R}, perm::Bool=false)
    nx = size(A, 1)
    # Compute a scaling of the system matrix M
    S = diag(balance([A B; C 0], false)[1])
    Sx = S[1:nx]
    Sio = S[nx+1]
    # Compute permutation of x (if requested)
    pvec = perm ? balance(A, true)[2] * collect(1:nx) : collect(1:nx)
    # Compute the transformation matrix
    T = zeros(R, nx, nx)
    T[pvec, :] = Sio * diagm(1./Sx)
    return T
end

function tf2ss(num::Array,den::Array)
  (mnum,nnum)  = size(num,1,2);
  (mden,n)     = size(den,1,2);

  if  (n == 0 && nnum == 0)
    a=[]
    b=[]
    c=[]
    d=[]
    return a,b,c,d
  end

  if min(mden,n)>1
    error("denominator needs to be normalized")
  elseif mden>1
    den = transpose(den);
  end

  inz = find(den);
  den = den[inz[1]:end];
  n   = length(den);

#=
% Check for proper numerator
if nnum > n
    % Try to strip leading zeros to make proper
    if (all(all(num(:,1:(nnum-n)) == 0)))
        num = num(:,(nnum-n+1):nnum);
        [mnum,nnum] = size(num);
    else
       ctrlMsgUtils.error('Controllib:general:DenomInvalidOrder');
    end
end
=#

# Pad numerator with leading zeros, to make it have the same number of
# columns as the denominator, and normalize it to den(1)
  println(typeof(n))
  println(typeof(nnum))
  num = [zeros(mnum,n-nnum) num]./den[1];

  # Do the D-matrix first
  if ~isempty(num)
      d = num[:,1];
  else
      d = [];
  end

  # Handle special constant case:
  if n == 1
      a = [];
      b = [];
      c = [];
      return a,b,c
  end

  # Now do the rest, starting by normalizing den to den(1),
  den = transpose(den[2:n] ./ den[1]);
  a = [-den; eye(n-2,n-1)];
  b = eye(n-1,1);
  if mnum > 0
      c = num[:,2:n] - num[:,1] * den;
  else
      c = [];
  end

  return a,b,c,d
end
