typealias ZPKMatrix Matrix{Vector{Complex128}}

type ZPK <: LTISystem
    z::ZPKMatrix
    p::ZPKMatrix
    k::Array{Float64, 2}
    Ts::Float64
    ny::Int
    nu::Int
    inputnames::Vector{UTF8String}
    outputnames::Vector{UTF8String}
    function ZPK(z::ZPKMatrix, p::ZPKMatrix, k::Matrix{Float64}, Ts::Float64, inputnames::Vector{UTF8String}, outputnames::Vector{UTF8String})
        ny, nu = size(z)

        # validate matrix dimensions and size of input/output names
        if size(p) != (ny,nu) || size(k) != (ny,nu)
            error("Dimesion mishmatch in zpk matrices")
        elseif size(inputnames, 1) != nu
            error("Must have same number of inputnames as inputs")
        elseif size(outputnames, 1) != ny
            error("Must have same number of outputnames as outputs")
        end

        # Validate sampling time
        if Ts < 0 && Ts != -1
            error("Ts must be either a positive number, 0 (continuous system), or -1 (unspecified)")
        end
        return new(z, p, k, Ts, ny, nu, inputnames, outputnames)
    end
end

#####################################################################
##                          Constructors                           ##
#####################################################################

function zpk(z::ZPKMatrix, p::ZPKMatrix, k::Matrix{Float64}, Ts::Float64=0.0; kwargs...)
    ny, nu = size(z)

    kvs = Dict(kwargs)
    inputnames = validate_names(kvs, :inputnames, nu)
    outputnames = validate_names(kvs, :outputnames, ny)
    return ZPK(z, p, k, Float64(Ts), inputnames, outputnames)
end
zpk{T1<:Number, T2<:Number, T3<:Real}(z::Matrix{Vector{T1}}, p::Matrix{Vector{T2}}, k::Matrix{T3}, args...) = zpk(ZPKMatrix(z), ZPKMatrix(p), map(Float64, k), args...)
zpk(z::AbstractVector, p::AbstractVector, k::Real, args...) = zpk(fill!(ZPKMatrix(1,1), z), fill!(ZPKMatrix(1,1), p), fill!(Array(Float64, 1,1), k), args...)

# zpk(sys) converts to ZPK
zpk(sys::LTISystem) = convert(ZPK, sys)
zpk(sys::ZPK) = sys

# Function for creation of static gain
function zpk(gain::Array, Ts::Real=0.0; kwargs...)
    ny, nu = size(gain, 1, 2)
    zp = [Array(Complex128, 0) for i=1:ny, j=1:nu]

    kvs = Dict(kwargs)
    inputnames = validate_names(kvs, :inputnames, nu)
    outputnames = validate_names(kvs, :outputnames, ny)
    return ZPK(zp, zp, [Float64(gain[i,j]) for i=1:ny, j=1:nu], Float64(Ts), inputnames, outputnames)
end
zpk(gain::Real, Ts::Real=0.0; kwargs...) = zpk([gain], Ts, kwargs...)

#####################################################################
##                          Conversions                            ##
#####################################################################

# LTI -> ZPK
Base.convert(::Type{ZPK}, sys::LTISystem) = zpk(zpkdata(sys)..., sys.Ts, inputnames=sys.inputnames, outputnames=sys.outputnames)

# ZPK -> transferfunction
function Base.convert(::Type{TransferFunction}, sys::ZPK)
    z,p,k = zpkdata(sys)
    ny, nu = size(z)
    matrix = [k[i,j]*SisoTf(real(poly(z[i,j])), real(poly(p[i,j]))) for i=1:ny, j=1:nu]
    return TransferFunction(matrix, sys.Ts, sys.inputnames, sys.outputnames)
end

# Numbers -> ZPK
Base.convert(::Type{ZPK}, A::Array) = zpk(A)
Base.convert(::Type{ZPK}, n::Real) = zpk(n)

### Promotions ###
Base.promote_rule(::Type{StateSpace}, ::Type{ZPK}) = StateSpace
Base.promote_rule(::Type{TransferFunction}, ::Type{ZPK}) = TransferFunction

### Helper Conversions ###
# Convert a matrix of vectors to a matrix of complex vectors
function Base.convert(::ZPKMatrix, x::Matrix{Vector})
    ny, nu = size(x)
    xzpk = ZPKMatrix(ny, nu)
    for i=1:ny, j=1:nu
        xzpk[i,j] = map(Complex128, x[i,j])
    end
    return xzpk
end
# To make blkdiag for ZPKMatrix possible:
Base.zero(::Type{Vector{Complex128}}) = Vector{Complex128}(0)

#####################################################################
##                          Misc. Functions                        ##
#####################################################################

## INDEXING ##
Base.ndims(t::ZPK) = 2
Base.size(t::ZPK) = t.ny, t.nu
Base.size(t::ZPK, d) = d <= 2 ? size(t)[d] : 1

function Base.getindex(t::ZPK, inds...)
    rows, cols = checkindex(t, collect(inds))
    ny, nu = length(rows), length(cols)
    z = Array(Vector{Complex128}, ny, nu)
    p = copy(z)
    k = Array(Float64, ny, nu)

    for i=1:ny, j=1:nu
        z[i,j] = t.z[rows[i], cols[j]]
        p[i,j] = t.p[rows[i], cols[j]]
        k[i,j] = t.k[rows[i], cols[j]]
    end
    return ZPK(z, p, k, t.Ts, vcat(t.inputnames[cols]), vcat(t.outputnames[rows]))
end

function Base.setindex!(t1::ZPK, t2::ZPK, inds...)
    if size(inds, 1) != 2
        error("Must specify 2 indices to index TransferFunction model")
    end
    rows, cols = checkindex(t1, collect(inds))
    ny, nu = length(rows), length(cols)
    for i=1:ny
        t1.outputnames[rows[i]] = t2.outputnames[i]
        for j=1:nu
            t1.z[rows[i], cols[j]] = t2.z[i,j]
            t1.p[rows[i], cols[j]] = t2.p[i,j]
            t1.k[rows[i], cols[j]] = t2.k[i,j]
        end
    end
    for j = 1:nu
        t1.inputnames[cols[j]] = t2.inputnames[j]
    end

    return t1
end

function checkindex(sys::LTISystem, inds::Vector)
    # Check dimensions
    if size(inds, 1) != 2
        error("Must specify 2 indices to index TransferFunction model")
    end

    # Handle case where one or more indecies are colons (":")
    rows = inds[1] == Colon() ? (1:sys.ny) : inds[1]
    cols = inds[2] == Colon() ? (1:sys.nu) : inds[2]
    return rows, cols
end

# Doesn't work (since transposing Matrix{Vector} isn't supported in Julia... would need to implement it)
# Base.transpose(t::ZPK) = ZPK(t.z', t.p', t.k', t.Ts, t.outputnames, t.inputnames)
# Base.ctranspose(t::ZPK) = transpose(t)

function Base.copy(t::ZPK)
    z, p, k, inputnames, outputnames = map(copy, (zpkdata(t)..., t.inputnames, t.outputnames))
    return ZPK(z, p, k, t.Ts, inputnames, outputnames)
end

#####################################################################
##                         Math Operators                          ##
#####################################################################

## IDENTITIES ##
Base.one(::ZPK) = zpk(1)
Base.one(::Type{ZPK}) = zpk(1)
Base.zero(::ZPK) = zpk(0)
Base.zero(::Type{ZPK}) = zpk(0)

## EQUALITY ##
function ==(t1::ZPK, t2::ZPK)
    fields = [:Ts, :inputnames, :outputnames, :z, :p, :k]
    for field in fields
        if getfield(t1, field) != getfield(t2, field)
            return false
        end
    end
    return true
end

## ADDITION ##
function +(t1::ZPK, t2::ZPK)
    if size(t1) != size(t2)
        error("Systems have different shapes.")
    elseif t1.Ts != t2.Ts
        error("Sampling time mismatch")
    end
    ny, nu = size(t1)

    # Naming strategy: If only one sys is named, use that. If the names are the
    # same, use them. If the names conflict, then they are ignored, and the
    # default "" is used.
    if all(t1.inputnames .== "")
        inputnames = t2.inputnames
    elseif all(t2.inputnames .== "") || (t1.inputnames == t2.inputnames)
        inputnames = t1.inputnames
    else
        inputnames = UTF8String["" for i = 1:ny]
    end
    if all(t1.outputnames .== "")
        outputnames = t2.outputnames
    elseif all(t2.outputnames .== "") || (t1.outputnames == t2.outputnames)
        outputnames = t1.outputnames
    else
        outputnames = UTF8String["" for i = 1:nu]
    end

    z = Array(Vector{Complex128}, ny, nu)
    p = copy(z)
    k = Array(Float64, ny, nu)

    for i=1:ny, j=1:nu
        Z = t1.k[i,j]*poly(t1.z[i,j])*poly(t2.p[i,j]) + t2.k[i,j]*poly(t2.z[i,j])*poly(t1.p[i,j])
        z[i,j] = roots(Z)
        p[i,j] = vcat(t1.p[i,j], t2.p[i,j])
        k[i,j] = Z[1]
    end
    return ZPK(z, p, k, t1.Ts, inputnames, outputnames)
end
+(t::ZPK, n::Real) = +(t, zpk(n*ones(size(t)), t.Ts))
+(n::Real, t::ZPK) = +(t, n)

## NEGATION ##
-(t::ZPK) = ZPK(t.z, t.p, -t.k, t.Ts, t.inputnames, t.outputnames)
-(t::ZPK, n::Real) = +(t1, -n)
-(n::Real, t::ZPK) = +(n, -t)

function *(t1::ZPK, t2::ZPK)
    # Note: t1*t2 = y <- t1 <- t2 <- u
    if t1.nu != t2.ny
        error("t1*t2: t1 must have same number of inputs as t2 has outputs")
    elseif t1.Ts != t2.Ts
        error("Sampling time mismatch")
    end
    ny, nu = size(t1, 1), size(t2, 2)

    # Initialize variables using a ZPK model
    t = ZPK([Vector{Complex128}(0) for i=1:ny, j=1:nu], [Vector{Complex128}(0) for i=1:ny, j=1:nu], zeros(ny, nu), t1.Ts, t2.inputnames, t1.outputnames)

    for i=1:ny, j=1:nu, a=1:size(t1, 2)
        _zpk_add!(t, _zpk_mult(t1, t2, i, a, j)..., i, j)
    end

    t.inputnames = t2.inputnames

    return t
end

# multiplies two sets of siso ZPK data (for internal use)
function _zpk_mult(t1::ZPK, t2::ZPK, i::Int, a::Int, j::Int)
    return vcat(t1.z[i,a], t2.z[a,j]), vcat(t1.p[i,a], t2.p[a,j]), t1.k[i,a]*t2.k[a,j]
end

# adds two sets of siso ZPK data by cross multiplication
function _zpk_add!(t1::ZPK, z2::Vector{Complex128}, p2::Vector{Complex128}, k2::Float64, i::Int, j::Int)
    # Build numerator polynomial
    Z = t1.k[i,j]*poly(vcat(t1.z[i,j], p2)) + k2*poly(vcat(z2, t1.p[i,j]))

    # Calculate new roots
    t1.z[i,j] = roots(Z)

    # Multiply denominator polynomials
    append!(t1.p[i,j], p2)

    # Get new gain
    t1.k[i,j] = real(Z[1])

    return t1
end

### PUT IN lti.jl (and add convert(LTISystem, Array) to other types)###
function *(A::Array, sys::LTISystem)
    Asys = convert(typeof(sys), A)
    Asys.Ts = sys.Ts
    return Asys*sys
end
function *(sys::LTISystem, A::Array)
    Asys = convert(typeof(sys), A)
    Asys.Ts = sys.Ts
    return sys*Asys
end

*(t::ZPK, n::Real) = ZPK(t.z, t.p, n*t.k, t.Ts, t.inputnames, t.outputnames)
*(n::Real, t::ZPK) = *(t, n)


## DIVISION ##
function /(n::Real, t::ZPK)
    !issiso(t) && error("MIMO ZPK inversion isn't implemented yet")
    return ZPK(t.p, t.z, n./t.k, t.Ts, t.outputnames, t.inputnames)
end
/(t::ZPK, n::Real) = t*(1/n)
/(t1::ZPK, t2::ZPK) = t1*(1/t2)

#####################################################################
##                        Display Functions                        ##
#####################################################################

Base.print(io::IO, t::ZPK) = show(io, t)

function Base.show(io::IO, t::ZPK)
    # Compose the name vectors
    inputs = format_names(t.inputnames, "Input ", "?")
    outputs = format_names(t.outputnames,   "Output ", "?")
    println(io, "ZPK:")
    var = iscontinuous(t) ? :s : :z
    for i=1:t.ny, j=1:t.nu
        if !issiso(t)
            println(io, inputs[j], " to ", outputs[i])
        end
        print_sisozpk(io, t[i, j], var)
        if !(j == t.nu && i == t.ny)
            print(io, "\n")
        end
    end
    if iscontinuous(t)
        print(io, "\nContinuous-time zero-pole-gain model")
    else
        print(io, "\nSample Time: ")
        if t.Ts > 0
            print(io, t.Ts, " (seconds)")
        elseif t.Ts == -1
            print(io, "unspecified")
        end
        print(io, "\nDiscrete-time zero-pole-gain model")
    end
end

function print_sisozpk(io::IO, t::ZPK, var=:s)
    numstr = sprint(print_polyroots, t.z[1], var)
    denstr = sprint(print_polyroots, t.p[1], var)
    gainstr = t.k[1]==1.0 ? "" : "$(round(t.k[1], 6))"

    # Figure out the length of the separating line
    len_num = length(numstr)
    len_den = length(denstr)
    len_gain = length(gainstr)
    dashcount = max(len_num, len_den)

    # Center the numerator or denominator
    if len_num < dashcount
        numstr = "$(repeat(" ", div(dashcount - len_num, 2)))$numstr"
    else
        denstr = "$(repeat(" ", div(dashcount - len_den, 2)))$denstr"
    end
    println(io, repeat(" ", len_gain+1), numstr)
    println(io, gainstr, " ", repeat("-", dashcount))
    println(io, repeat(" ", len_gain+1), denstr)
end


# PUT THIS IN poly.jl
function print_polyroots(io::IO, z::Vector{Complex128}, var=:x)
    z = z[imag(z) .>= -abs(z)*sqrt(eps(Float64))]
    n = length(z)
    if n == 0
        print(io, "1.0")
    else
        j = 1
        while length(z) != 0
            zj = z[j]
            tol = abs(zj) * sqrt(eps(Float64))
            if abs(zj) >= 2*eps(Float64)
                sgn = real(zj) < 0 ? (:+) : (:-)
                if imag(zj) >= tol
                    tmp = abs(round(2*real(zj), 6))
                    if tmp == 0
                        str = "($(var)^2 + $(round(abs(zj)^2, 6)))"
                    else
                        str = "($(var)^2 $sgn $tmp$var + $(round(abs(zj)^2, 6)))"
                    end
                else
                    str = "($var $sgn $(round(abs(real(zj)), 6)))"
                end
            else
                str = "$var"
            end
            inds = find(x->abs(x-zj) <= 2*tol, z)
            deleteat!(z, inds)
            exp = length(inds)
            if exp == 1
                print(io, str)
            else
                print(io, str, '^', exp)
            end
        end
    end
end
