# Misfit functions

## Predicates ##

@doc """`isstable(sys)` or `isstable(poles)`

Returns `true` if `sys` is stable, else returns `false`.""" ->
function isstable(ps::Vector{Float64}, continous::Bool=true)
    # inputs are real
    return !any(continous ? ps .> 0 : abs(ps) .> 1)
end
function isstable(sys::LTISystem)
    return isstable((sys.Ts==0 ? real:abs)(pole(sys)), sys.Ts==0)
end
function isstable(ps::Vector{Complex128}, continous::Bool=true)
    # Complex parts aren't needed to determine stability
    return isstable((continous ? real:abs)(ps), continous)
end

# if continous
#     return !any(ps .> 0)
# else
#     return !any(abs(ps).>=1)
# end

@doc """`iscontinuous(sys)`

Returns `true` if `sys` is continuous, else returns `false`.""" ->
function iscontinuous(sys::LTISystem)
    return sys.Ts == 0
end

@doc """`issiso(sys)`

Returns `true` if `sys` is SISO, else returns `false`.""" ->
function issiso(sys::LTISystem)
    return sys.ny == 1 && sys.nu == 1
end

@doc """`isproper(tf)`

Returns `true` if the `TransferFunction` is proper. This means that order(den) >= order(num))""" ->
function isproper(t::TransferFunction)
    for s in t.matrix
        if length(s.num) > length(s.den)
            return false
        end
    end
    return true
end
function isproper(sys::ZPK)
    ny, nu = size(sys)
    for i=1:ny, j=1:nu
        if length(sys.z[i,j]) > length(sys.p[i,j])
            return false
        end
    end
    return true
end

@doc """`A, B, C, D = ssdata(tf)`

Returns the system matrices for state space system `sys`""" ->
function ssdata(sys::StateSpace)
    return sys.A, sys.B, sys.C, sys.D
end
ssdata(sys::LTISystem) = ssdata(ss(sys))

@doc """`order(sys)`

Returns the order of the `LTISystem` `sys`.""" ->
function order(sys::TransferFunction)
    ny, nu = size(sys)
    n = 1
    for i=1:ny, j=1:nu
        n = max(n, length(sys.matrix[i,j].num), length(sys.matrix[i,j].den))
    end
    return n - 1
end
function order(sys::ZPK)
    ny, nu = size(sys)
    n = 0
    for i=1:ny, j=1:nu
        n = max(n, length(sys.z[i,j]), length(sys.p[i,j]))
    end
    return n
end
order(sys::StateSpace) = size(sys.A, 1)


## Helpers (not public) ##

# Convert the argument to a Matrix{Float64}
function float64mat(A::Vector)
    A = reshape(A, size(A, 1), 1)
    return float64mat(A)
end
float64mat(A::Matrix) = map(Float64, A)
float64mat(A::Matrix{Float64}) = A

# Ensures the metadata for an LTISystem is valid
function validate_names(kwargs, key, n)
    names = get(kwargs, key, "")
    if names == ""
        return UTF8String[names for i = 1:n]
    elseif isa(names, Vector) && eltype(names) <: UTF8String
        return names
    elseif isa(names, UTF8String)
        return UTF8String[names * "$i" for i = 1:n]
    else
        error("$key must be of type `UTF8String` or Vector{UTF8String}")
    end
end

# Format the metadata for printing
function format_names(names::Vector{UTF8String}, default::String, unknown::String)
    n = size(names, 1)
    if all(names .== "")
        return UTF8String[default * string(i) for i=1:n]
    else
        for (i, n) in enumerate(names)
            names[i] = (n == "") ? unknown : n
        end
        return names
    end
end

# Unwrap function (by ssfrr at https://gist.github.com/ssfrr/7995008)
@doc """phaseunwrap(phase; rad=true/false)

Unwraps the phase vector `phase` by adding multiples of 2π/360 when there is a skip between entries larger than π/180. By default assumes the phase data is in degrees.""" ->
function phaseunwrap(v::Vector{Float64}, inplace=false; rad::Bool=false)
  unwrapped = inplace ? v : copy(v)
  halfturn = rad ? pi : 180
  for i in 2:length(v)
    while unwrapped[i] - unwrapped[i-1] >= halfturn
      unwrapped[i] -= 2*halfturn
    end
    while unwrapped[i] - unwrapped[i-1] <= -halfturn
      unwrapped[i] += 2*halfturn
    end
  end
  return unwrapped
end
@doc """phaseunwrap!(phase; rad=true/false)

Unwraps the phase vector `phase` by adding multiples of 2π/360 when there is a skip between entries larger than π/180. By default assumes the phase data is in degrees.""" ->
phaseunwrap!(v; rad::Bool=false) = phaseunwrap(v, true, rad=rad)
