
open Ustring.Op
open Info

(** Handling of Modelica messages (errors and warnings) *)

type id =
  | LEX_UNKNOWN_CHAR
      (* arg1 - String representing the unknown char *)
  | LEX_COMMENT_NOT_TERMINATED
      (* arg1 = The comment string *)
  | PARSE_ERROR
      (* *)
  | VARIABLE_CAPTURED
      (* arg1 = The variable that is captured *)

type severity =
  | ERROR
  | WARNING

type arguments = ustring list

(** Error and warning messages. Created by the lexer, parser,
    and type checker.  *)
type message = id * severity * info * arguments

(** [id2str id] returns the identifier string for [id], e.g., 
    "LEX_UNKNOWN_CHAR" *)
let id2str id =
  match id  with
    | LEX_UNKNOWN_CHAR -> us"Unknown character"
    | LEX_COMMENT_NOT_TERMINATED -> us"Comment is not terminated"
    | PARSE_ERROR -> us"Parse error"
    | VARIABLE_CAPTURED -> us"Variable captured when substituting"

(** [severity2str s] returns the severity strings ["ERROR"] or 
    ["WARNING"]. *)
let severity2str s =
  match s with
    | ERROR -> us"ERROR"
    | WARNING -> us"WARNING"

let info2str_startline fi =
  match fi with 
    | Info(filename,l1,c1,l2,c2) -> l1
    | NoInfo -> assert false
  
(** [message2str m] returns a string representation of message [m].
    Is message is not intended to be read by humans. *)
let message2str (id,sev,info,args)  = 
  match info with 
    | Info(filename,l1,c1,l2,c2) -> 
	begin
	  let sargs = us"[" ^. (List.fold_left 
	    (fun a b -> a ^. (if a =. us"" then us"" else us",") ^. b)
	   (us"") args) ^. us"]" 
	in
	  us"FILE \"" ^. filename ^. us"\" " ^.
	    (ustring_of_int l1) ^. us":" ^.
	    (ustring_of_int c1) ^. us"-" ^.
	    (ustring_of_int l2) ^. us":" ^.
	    (ustring_of_int c2) ^. us" " ^.
	    (severity2str sev) ^. us": " ^. 
	    (id2str id) ^. us" " ^. sargs 
        end
    |  NoInfo -> us"NO INFO: " ^. (id2str id)  
     






