From TLC Require Import LibEnv LibTactics.

Inductive expr :=
| ENat (n: nat)
| EVar (x: var)
| EAdd (e1 e2: expr)
| ESub (e1 e2: expr)
| EMul (e1 e2: expr)
| EIfZero (cnd thn els: expr).

Inductive program :=
| PLet (x: var) (e: expr) (p: program)
| PRes (e: expr).

Inductive wf_expr (vars: vars) : expr -> Prop :=
| WFNat: forall n, wf_expr vars (ENat n)
| WFVar: forall x, x \in vars -> wf_expr vars (EVar x)
| WfAdd: forall e1 e2,
    wf_expr vars e1 ->
    wf_expr vars e2 ->
    wf_expr vars (EAdd e1 e2)
| WfSub: forall e1 e2,
    wf_expr vars e1 ->
    wf_expr vars e2 ->
    wf_expr vars (ESub e1 e2)
| WfMul: forall e1 e2,
    wf_expr vars e1 ->
    wf_expr vars e2 ->
    wf_expr vars (EMul e1 e2)
| WfIfZero: forall cnd thn els,
    wf_expr vars cnd ->
    wf_expr vars thn ->
    wf_expr vars els ->
    wf_expr vars (EIfZero cnd thn els).

Inductive wf_program' : vars -> program -> Prop :=
| WFLet: forall vars x e p,
    wf_expr vars e ->
    x \notin vars ->
    wf_program' (\{x} \u vars) p ->
    wf_program' vars (PLet x e p)
| WfResult: forall max e,
    wf_expr max e ->
    wf_program' max (PRes e).

Definition wf_program := wf_program' \{}.

Fixpoint eval_expr (env: env nat) (e: expr): option nat :=
  match e with
  | ENat n => Some n
  | EVar x => get x env
  | EAdd e1 e2 =>
      match eval_expr env e1, eval_expr env e2 with
      | Some n1, Some n2 => Some (n1 + n2)
      | _, _ => None
      end
  | ESub e1 e2 =>
      match eval_expr env e1, eval_expr env e2 with
      | Some n1, Some n2 => Some (n1 - n2)
      | _, _ => None
      end
  | EMul e1 e2 =>
      match eval_expr env e1, eval_expr env e2 with
      | Some n1, Some n2 => Some (n1 * n2)
      | _, _ => None
      end
  | EIfZero cnd thn els =>
      match eval_expr env cnd with
      | Some 0 => eval_expr env thn
      | Some _ => eval_expr env els
      | None => None
      end
  end.

Fixpoint eval_program' (env: env nat) (p: program) : option nat :=
  match p with
  | PLet x e p' =>
      match eval_expr env e with
      | Some n => eval_program' (env & x ~ n) p'
      | None => None
      end
  | PRes e => eval_expr env e
  end.

Definition eval_program p := eval_program' empty p.

Ltac auto_tilde ::=
  try (solve [ auto with lang
             | intuition auto with lang ]).

Ltac auto_star ::=
  try (solve [ eauto with lang
             | jauto
             | intuition eauto with lang ]).

Ltac cases_match :=
  match goal with
  | _ : _ |- context[match ?n with | _ => _ end] => destruct n
  end.

Tactic Notation "cases_match" "~" := cases_match; auto_tilde.
Tactic Notation "cases_match" "*" := cases_match; auto_star.

Hint Extern 2 => cases_match : lang.

(* Theorems *)
Lemma wf_expr_eval:
  forall e env,
    wf_expr (dom env) e ->
    exists n, eval_expr env e = Some n.
Proof.
  introv Hwf.
  induction Hwf; simpls*.
  - apply~ get_some.
Qed.

Local Hint Resolve wf_expr_eval : lang.

Lemma wf_program'_eval:
  forall env p,
    wf_program' (dom env) p ->
    exists n, eval_program' env p = Some n.
Proof.
  introv Hwf.
  inductions Hwf; simpls*.
  forwards* (n & ->): wf_expr_eval.
  forwards*: IHHwf. simpl_dom; autos*.
Qed.

Theorem wf_program_eval:
  forall p,
    wf_program p ->
    exists n, eval_program p = Some n.
Proof.
  introv Hwf.
  unfolds wf_program.
  forwards*: wf_program'_eval.
  simpl_dom. autos.
Qed.