Require Import List.
Import List.ListNotations.

Definition var : Type := nat.

Inductive expr :=
| ENat (n: nat)
| EVar (x: var)
| EAdd (e1 e2: expr)
| EMul (e1 e2: expr)
| EIfZero (cnd thn els: expr).

Inductive program :=
| PLet (e: expr) (p: program)
| PRes (e: expr).

Inductive wf_expr (max: var) : expr -> Prop :=
| WFNat: forall n, wf_expr max (ENat n)
| WFVar: forall x, x < max -> wf_expr max (EVar x)
| WfAdd: forall e1 e2,
    wf_expr max e1 ->
    wf_expr max e2 ->
    wf_expr max (EAdd e1 e2)
| WfMul: forall e1 e2,
    wf_expr max e1 ->
    wf_expr max e2 ->
    wf_expr max (EMul e1 e2)
| WfIfZero: forall cnd thn els,
    wf_expr max cnd ->
    wf_expr max thn ->
    wf_expr max els ->
    wf_expr max (EIfZero cnd thn els).

Inductive wf_program' : var -> program -> Prop :=
| WFLet: forall max e p,
    wf_expr max e ->
    wf_program' (S max) p ->
    wf_program' max (PLet e p)
| WfResult: forall max e,
    wf_expr max e ->
    wf_program' max (PRes e).

Definition wf_program := wf_program' 0.

Definition environment := list nat.

Fixpoint eval_expr (env: environment) (e: expr): option nat :=
  match e with
  | ENat n => Some n
  | EVar x => nth_error env x
  | EAdd e1 e2 =>
      match eval_expr env e1, eval_expr env e2 with
      | Some n1, Some n2 => Some (n1 + n2)
      | _, _ => None
      end
  | EMul e1 e2 =>
      match eval_expr env e1, eval_expr env e2 with
      | Some n1, Some n2 => Some (n1 * n2)
      | _, _ => None
      end
  | EIfZero cnd thn els =>
      match eval_expr env cnd with
      | Some 0 => eval_expr env thn
      | Some _ => eval_expr env els
      | None => None
      end
  end.

Fixpoint eval_program' (env: environment) (p: program) : option nat :=
  match p with
  | PLet e p' =>
      match eval_expr env e with
      | Some n => eval_program' (env ++ [n]) p'
      | None => None
      end
  | PRes e => eval_expr env e
  end.

Definition eval_program p := eval_program' nil p.

(* Theorems *)
Lemma wf_expr_eval:
  forall e env,
    wf_expr (length env) e ->
    exists n, eval_expr env e = Some n.
Proof.
  intros e env Hwf.
  induction Hwf;
    simpl; eauto;
    try solve [
        repeat
          match goal with
          | H : exists n, _ = _ |- _ => destruct H as [? ->]
          end; eauto
      ].
  - apply nth_error_Some in H.
    assert (Hex: exists n, nth_error env x = Some n).
    {
      destruct (nth_error env x) as [n |]; eauto.
      - congruence.
    }
    destruct Hex as [n Heq].
    eauto.
  - destruct IHHwf1 as [n1 ->].
    destruct IHHwf2 as [n2 ->].
    destruct IHHwf3 as [n3 ->].
    match goal with
    | _ : _ |- context[match ?n with | _ => _ end] => destruct n
    end; eauto.
Qed.

Lemma wf_program'_eval:
  forall env p,
    wf_program' (length env) p ->
    exists n, eval_program' env p = Some n.
Proof.
  intros env p Hwf.
  remember (length env).
  generalize dependent env.
  induction Hwf.
  - intros env Heq. subst.
    simpl. apply wf_expr_eval in H.
    destruct H.
    rewrite H.
    assert (Hlen: S (length env) = length (env ++ x :: nil)).
    {
      rewrite app_length. simpl. rewrite PeanoNat.Nat.add_1_r.
      reflexivity.
    }
    apply IHHwf in Hlen.
    destruct Hlen.
    exists x0. apply H0.
  - intros env Heq. subst.
    simpl. apply wf_expr_eval in H.
    apply H.
Qed.

Theorem wf_program_eval:
  forall p,
    wf_program p ->
    exists n, eval_program p = Some n.
Proof.
  intros p Hwf.
  apply wf_program'_eval.
  apply Hwf.
Qed.
