From TLC Require Import LibLN LibEnv LibTactics.

Ltac auto_tilde ::=
  try (solve [ auto with stlc
             | intuition auto with stlc ]).

Ltac auto_star ::=
  try (solve [ eauto with stlc
             | jauto
             | intuition eauto with stlc ]).

Inductive type :=
  | TArr : type -> type -> type
  | TBase : type
.

Inductive term :=
  | TmBVar : nat -> term
  | TmFVar : var -> term
  | TmAbs : type -> term -> term
  | TmApp : term -> term -> term
  | TmConst : term
.

(* Opening a term replaces a bound variable by some other term. *)
Reserved Notation "{ k ~> u } t" (at level 67).
Fixpoint open' (k : nat) (u : term) (t : term) :=
  match t with
  | TmBVar i => If i = k then u else t
  | TmFVar y => t
  | TmAbs T t' => TmAbs T ({(S k) ~> u} t')
  | TmApp t1 t2 => TmApp ({k ~> u}t1) ({k ~> u}t2)
  | TmConst => t
  end
where "{ k ~> u } t " := (open' k u t)
.

Definition open := open' 0.
Notation " t ^^ u " := (open u t) (at level 67).
Notation " t ^ x " := (open (TmFVar x) t).

Inductive lc : term -> Prop :=
  | LCFVar : forall x, lc (TmFVar x)
  | LCAbs : forall L t T, (forall x, x \notin L -> lc (t^x)) -> lc (TmAbs T t)
  | LCApp : forall t1 t2, lc t1 -> lc t2 -> lc (TmApp t1 t2)
  | LCConst : lc TmConst
.
Hint Constructors lc : stlc.

Inductive value : term -> Prop :=
  | ValAbs : forall T t, lc (TmAbs T t) -> value (TmAbs T t)
  | ValConst : value TmConst
.
Hint Constructors value : stlc.

Fixpoint fv (t : term) :=
  match t with
  | TmBVar i => \{}
  | TmFVar x => \{x}
  | TmAbs T t' => fv t'
  | TmApp t1 t2 => fv t1 \u fv t2
  | TmConst => \{}
  end
.

Reserved Notation "[ x ~> u ] t" (at level 67).
Fixpoint subst (x : var) (u : term) (t : term) :=
  match t with
  | TmBVar i => t
  | TmFVar y => If x = y then u else t
  | TmAbs T t' => TmAbs T ([x ~> u] t')
  | TmApp t1 t2 => TmApp ([x ~> u]t1) ([x ~> u]t2)
  | TmConst => t
  end
where "[ x ~> u ] t" := (subst x u t).

Reserved Notation " t '-->' t' " (at level 50).
Inductive step : term -> term -> Prop :=
  | StepApp1 : forall t1 t1' t2,
      t1 --> t1' ->
      TmApp t1 t2 --> TmApp t1' t2
  | StepApp2 : forall v1 t2 t2',
      value v1 ->
      t2 --> t2' ->
      TmApp v1 t2 --> TmApp v1 t2'
  | StepAppAbs : forall T t1 v2,
      value v2 ->
      TmApp (TmAbs T t1) v2 --> (t1^^v2)
  where " t1 '-->' t2 " := (step t1 t2)
.
Hint Constructors step : stlc.

Inductive multistep : term -> term -> Prop :=
  | MStepRefl : forall t, multistep t t
  | MStepTrans : forall t1 t2 t3, t1 --> t2 -> multistep t2 t3 -> multistep t1 t3.
Notation " t '-->*' t' " := (multistep t t') (at level 50).
Hint Constructors multistep : stlc.

Definition ctx := env type.

Reserved Notation "Gamma |= t ~: T " (at level 50).
Inductive has_type : ctx -> term -> type -> Prop :=
  | TyFVar : forall x T Gamma,
      ok Gamma ->
      binds x T Gamma ->
      Gamma |= TmFVar x ~: T
  | TyAbs : forall L t2 T1 T2 Gamma,
      (forall x, x \notin L ->
                 (Gamma & x ~ T1) |= t2^x ~: T2) ->
      Gamma |= TmAbs T1 t2 ~: TArr T1 T2
  | TyApp : forall t1 t2 T1 T2 Gamma,
      Gamma |= t1 ~: TArr T1 T2 ->
      Gamma |= t2 ~: T1 ->
      Gamma |= TmApp t1 t2 ~: T2
  | TyConst : forall Gamma, ok Gamma -> Gamma |= TmConst ~: TBase
  where " Gamma |= t ~: T " := (has_type Gamma t T)
.
Hint Constructors has_type ok : stlc.

(* ================================================================ *)
(*   Tactics from Arthur Chargueraud's work on Locally Nameless     *)
(* ================================================================ *)

(** When picking a fresh atom or applying a rule that uses cofinite
    quantification, choosing a set of atoms to be fresh for can be
    tedious.  In practice, it is simpler to use a tactic to choose the
    set to be as large as possible.

    The first tactic we define, [gather_vars], is used to collect
    together all the atoms in the context.  It relies on an auxiliary
    tactic from [LibLN_Tactics], [gather_vars_with], which collects
    together the atoms appearing in objects of a certain type.  The argument
    to [gather_vars_with] is a function that should return the set of
    vars appearing in its argument.
*)

Ltac gather_vars :=
  let A := gather_vars_with (fun x : vars => x) in
  let B := gather_vars_with (fun x : var => \{x}) in
  let C := gather_vars_with (fun x : ctx => dom x) in
  let D := gather_vars_with (fun x : term => fv x) in
  constr:(A \u B \u C \u D).

(** The tactic [pick_fresh_gen L x] creates a new atom fresh
    from [L] and called [x]. Using the tactic [gather_vars],
    we can automate the construction of [L]. The tactic
    [pick_fresh x] creates a new atom called [x] that is fresh
    for "everything" in the context.
*)

Ltac pick_fresh x :=
  let L := gather_vars in (pick_fresh_gen L x).

(** The tactic [apply_fresh T as y] takes a lemma T of the form
    [forall L ..., (forall x, x \notin L, P x) -> ... -> Q.]
    and applies it by instantiating L as the set of variables
    occuring in the context (L is computed using [gather_vars]).
    Moreover, for each subgoal of the form [forall x, x \notin L, P x]
    being generated, the tactic automatically introduces the name [x]
    as well as the hypothesis [x \notin L].
*)

Tactic Notation "apply_fresh" constr(T) "as" ident(x) :=
  apply_fresh_base T gather_vars x.

(** The tactic [apply_fresh* T as y] is the same as
    [apply_fresh T as y] except that it calls [intuition eauto]
    subsequently. It is also possible to call [apply_fresh]
    without specifying the name that should be used.
*)

Tactic Notation "apply_fresh" "*" constr(T) "as" ident(x) :=
  apply_fresh T as x; intuition eauto.
Tactic Notation "apply_fresh" constr(T) :=
  apply_fresh_base T gather_vars ltac_no_arg.
Tactic Notation "apply_fresh" "*" constr(T) :=
  apply_fresh T; auto_star.

(* ================================================================ *)
(* ================================================================ *)

Lemma has_type_lc :
  forall Gamma t T,
    Gamma |= t ~: T ->
    lc t.
Proof.
  introv Htype.
  induction* Htype.
Qed.

Lemma has_type_ok :
  forall Gamma t T,
    Gamma |= t ~: T ->
    ok Gamma.
Proof.
  introv hasType.
  induction* hasType.
  pick_fresh_gen L x.
  forwards*: ok_push_inv x.
Qed.

Hint Extern 1 (ok ?Gamma) =>
  match goal with
  | H: has_type Gamma _ _ |- _ => apply has_type_ok
  end : stlc.

Hint Extern 1 (lc ?t) =>
  match goal with
  | H: has_type _ t _ |- _ => apply has_type_lc
  end : stlc.

Theorem progress :
  forall t T,
    empty |= t ~: T ->
    value t \/ exists t', t --> t'.
Proof.
  introv hasType.
  inductions hasType; substs*.
  - forwards*: binds_empty_inv.
  - left. constructors. apply LCAbs with (L := L).
    introv Hnotin. apply* has_type_lc.
  - right.
    forwards* [Hval1 | [t1' Hstep1]]: IHhasType1.
    forwards* [Hval2 | [t2' Hstep2]]: IHhasType2.
    inverts Hval1; inverts* hasType1.
Qed.

Lemma open_neq :
  forall j v i u t,
    i <> j ->
    {i ~> u}({j ~> v}t) = {j ~> v}t ->
    {i ~> u}t = t.
Proof.
  introv Hneq Heq. gen i j.
  induction *t; intros; inverts Heq; simpls; fequals*.
  - case_nat*. case_nat*.
Qed.

Lemma open_lc :
  forall k u t,
    lc t ->
    {k ~> u}t = t.
Proof.
  introv Hlc. gen k.
  induction Hlc; intros; simpls; fequals*.
  - unfolds open.
    pick_fresh x.
    rewrites* (open_neq 0 (TmFVar x)).
Qed.

Lemma subst_open_comm :
  forall x y u t,
    x <> y ->
    lc u ->
    ([x ~> u]t) ^ y = [x ~> u] (t ^ y).
Proof.
  introv Hneq Hlc.
  unfolds. generalize 0.
  induction *t; intros; simpls; repeat cases_if*; fequals*;
    eauto using open_lc.
  - unfolds subst. case_var*.
Qed.

Lemma subst_intro :
  forall x t u,
    x \notin (fv t) ->
    t ^^ u = [x ~> u](t ^ x).
Proof.
  introv Hfv.
  unfolds. generalize 0. induction t; intros; simpls; fequals*.
  - cases_if; simpls*.
    cases_if*.
  - case_var*.
Qed.

(* Order of Gs weird due to apply_empty tactic *)
Lemma has_type_weakening :
  forall G3 G2 G1 t T,
    G1 & G3 |= t ~: T ->
    ok (G1 & G2 & G3) ->
    G1 & G2 & G3 |= t ~: T.
Proof.
  introv hasType Hok. remember (G1 & G3) as H. gen G3.
  induction hasType; intros; substs*.
  - constructors*. apply* binds_weaken.
  - apply_fresh* TyAbs as y.
    apply_ih_bind* H0.
Qed.

Lemma has_type_subst :
  forall G2 G1 x T1 T2 t1 t2,
    G1 & x ~ T2 & G2 |= t1 ~: T1 ->
    G1 |= t2 ~: T2 ->
    G1 & G2 |= [x ~> t2]t1 ~: T1.
Proof.
  introv hasType1 hasType2.
  remember (G1 & x ~ T2 & G2) as G. gen G2.
  induction hasType1; intros; simpls; substs*.
  - case_var.
    + binds_get H0. apply_empty* has_type_weakening.
    + binds_cases H0; autos*.
  - apply_fresh* TyAbs as y.
    hint has_type_lc.
    rewrite* subst_open_comm. apply_ih_bind* H0.
Qed.

Theorem preservation :
  forall Gamma t t' T,
    Gamma |= t ~: T ->
    t --> t' ->
    Gamma |= t' ~: T.
Proof.
  introv hasType Hstep.
  gen t'.
  induction hasType; intros; inverts *Hstep.
  - inverts hasType1.
    pick_fresh x. rewrite * (subst_intro x).
    apply_empty* has_type_subst.
Qed.
