package se.kth.worker1;

import android.app.Activity;
import android.os.Bundle;
import android.os.Handler;
import android.view.Gravity;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;
import se.kth.worker1.R;

/** 
 * WorkerThreadExample1
 * An example on how to use a thread to run a heavy task in the background.
 * When the work is done a Handler is used to post the result to the
 * UI-thread (the only thread that is allowed to update the UI).
 * The heavy task (calculating primes) is defined in the inner class Worker 
 * (extends Thread).
 * The task to post to the UI-thread (update UI) is defined in UpdateUIOnWorkerFinished
 * (implements Runnable).
 * 
 * WorkerThreadExample2 uses anonymous inner classes instead when defining the tasks; 
 * less code, maybe a bit more confusing.
 */
public class WorkerActivity1 extends Activity {
	
	private Button calculateButton;
	private EditText textInput;
	private TextView textOutput;
	
	private Handler handler;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        
        calculateButton = (Button) findViewById(R.id.CalculateButton);
        textInput = (EditText) findViewById(R.id.InputEditText);
        textOutput = (TextView) findViewById(R.id.OutputTextView);
       
        // Handler created on UI-thread, hence posted Runnables will be
        // dispatched to that thread.
        handler = new Handler(); 
        
        calculateButton.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {
				calculateButton.setEnabled(false);
	    		String input = textInput.getText().toString();
	    		long limit = Long.parseLong(input);
				Worker worker = new Worker(limit);
				worker.start();
			}
        });
    }
    
    /**
     * This Runnable defines the task to be executed in the background:
     * Calculate number of primes and when this is done,
     * post an update to the UI-thread (using the handler).
     */
    private class Worker extends Thread {
    	
    	private long limit;
    	
    	public Worker(long limit) {
    		this.limit = limit;
    	}
    	
    	public void run() {
    		final long n = Prime.calculateNumberOfPrimes(limit);
    		final String output = 
    			"Number of primes <= " + limit + " equals " + n + EXCLAMATION;
    		
    		// The work is done, post back to UI-thread
    		handler.post(new UpdateUIOnWorkerFinished(output));
    	}
    }
    
    /**
     * This Runnable defines the task to post back to the UI-thread
     * when the worker has finished.
     */
    private class UpdateUIOnWorkerFinished implements Runnable {
    	
    	private String outputMessage;
    	
    	public UpdateUIOnWorkerFinished(String outputMessage) {
    		this.outputMessage = outputMessage;
    	}
    	public void run() {
			textOutput.setText(outputMessage);
			calculateButton.setEnabled(true);
			showToast("Worker thread finished");
    	}
    }
    
    private void showToast(String msg) {
    	Toast toast = Toast.makeText(this, msg, Toast.LENGTH_SHORT);
    	toast.setGravity(Gravity.CENTER, 0, 0);
    	toast.show();
    }
    
    private static final String EXCLAMATION = "\nWow, that's really interesting!";
}