/**
 * QR decoder class.
 * author: Claire
 * date: April 24th, 2013
 */

package se.kth.android.projectred.QR;

import java.util.Arrays;

import android.util.Log;

public class QRDecoderV2 {
	
	// Constants
	byte MAX_DATA = 14; // for error correction level High
	int SIZEVERSION = 25;

	// Variables
	ReedSolomon rs; // the Reed-Solomon decoder
	boolean[][] symbolReceived;
	boolean[][] symbolReceivedUnmasked;
	boolean[] codewordsStream; 
	int mode;
	int count;
	int lengthData;
	int size = SIZEVERSION;
	int nbSymb = 28;
	String outputMsg;
	byte[] dataOut;
	
	int formatDecoded = 0;
	int levelDecoded = 0;
	int maskDecoded = 0;
	
	int dataCodewords = 0;
	
	int nbErrors = 0;
	
	GenMask mask;
	GenFlag flag;
	
	public QRDecoderV2 (boolean[][] dataReceived) {
		
		rs = new ReedSolomon();
		symbolReceived = dataReceived;

		//Set flag and mask to version 2
		mask = new GenMask(2);
		flag = new GenFlag(2);
		
	}
	
	public String getMsg () {
		return this.outputMsg;
	}
	
	public void readFormat() { // Test ok, function correct!
		
		// Read format information
		
		boolean[] formatHorizontal = new boolean [15];
		boolean[] formatVertical = new boolean [15];
		boolean[] formatVerticalInv = new boolean [15];
		
		for (int col = 0 ; col < 6 ; col++) {
			int row = 8;
			formatHorizontal[col] = symbolReceived[row][col];
		}
		
		formatHorizontal[6] = symbolReceived[8][7];
		
		for (int col = 17 ; col < 25 ; col++) {
			int row = 8;
			formatHorizontal[col - 10] = symbolReceived[row][col];
		}
		
		for(int row = 0 ; row < 6 ; row++) {
			int col = 8;
			formatVerticalInv[row] = symbolReceived[row][col];
		}
		
		for (int row = 7; row < 9 ; row++){
			int col=8;
			formatVerticalInv[row-1] = symbolReceived[row][col];
		}
		
		for(int row = 18 ; row < 25 ; row++) {
			int col = 8;
			formatVerticalInv[row-10] = symbolReceived[row][col];
		}
		
		for(int i = 0; i < 15; i++){
			formatVertical[i] = formatVerticalInv[14 - i];
		}
		
		// Convert boolean arrays into byte arrays for XOR operations.
		
		byte[] formatHorizontalByte = booleanToByteArray(formatHorizontal);
		byte[] formatVerticalByte = booleanToByteArray(formatVertical);
		
		//Release the masking of format information
		
		formatHorizontalByte[0] ^= 255;
		formatHorizontalByte[1] ^= 255;
		
		formatVerticalByte[0] ^= 255;
		formatVerticalByte[1] ^= 255;
		
		byte[] maskFormat = {(byte)0x54, (byte)0x12};
		
		formatHorizontalByte[0] ^= maskFormat[0];
		formatHorizontalByte[1] ^= maskFormat[1];

		formatVerticalByte[0] ^= maskFormat[0];
		formatVerticalByte[1] ^= maskFormat[1];
		
		// Convert byte arrays into integer for decoding
		int formatHorizontalInt = byteArrayToInt(formatHorizontalByte);
		int formatVerticalInt = byteArrayToInt(formatVerticalByte);

		// Apply BCH decoding
		int formatHInt = rs.qrDecodeFormat(formatHorizontalInt);
		int formatVInt = rs.qrDecodeFormat(formatVerticalInt);
		
		if (formatHInt == formatVInt) {
			
			formatDecoded = formatVInt;
			levelDecoded = formatDecoded >> 3;
			maskDecoded = formatDecoded - (levelDecoded << 3);
			
			switch (levelDecoded) {
			case 0: // level 00 Middle
				dataCodewords = 28;
				break;
			case 1: // level 01 Low
				dataCodewords = 34;
				break;
			case 2: // level 10 High
				dataCodewords = 16;
				break;
			case 3: // level 11 Quality
				dataCodewords = 22;
				break;
			}		
		}
		else {
			Log.e("QRDecoder","Error format incorrect.");
		}	
	} // end readFormat
	
	// Remove the mask from the data
	public void removeMask () {

		for (int row = 0; row < size ; row ++){
			for (int col = 0 ; col < size ; col ++){
				if (flag.isAvailable (row, col)) {
					symbolReceived [row][col] = xor (symbolReceived[row][col] , mask.getElement (row, col, maskDecoded));
				}
			}
		}
	}
	
	/*
	 * Restores the codewordsStream (boolean array) from the QR matrix.
	 */
	public void restoreData () {
		
		//CONSTANTS to define up and down directions
		int up_left[] = {24, 20, 16, 12, 8, 3};
		int up_right[] = {25, 21, 17, 13, 9, 4};
		int up[] = {24, 20, 16, 12, 8, 3, 25, 21, 17, 13, 9, 4};
		int down_left[] = {22, 18, 14, 10, 5, 1};
		int down_right[] = {23, 19, 15, 11, 6, 2};
		int down[] = {22, 18, 14, 10, 5, 1, 23, 19, 15, 11, 6, 2};
		
		//VARIABLES
		int row = size;
		int column = size;
		int flagStream = 0;
		codewordsStream = new boolean [8*44];
		
		//Sort Arrays to allow for binary search
		Arrays.sort(up_left);
		Arrays.sort(up_right);
		Arrays.sort(up);
		Arrays.sort(down_left);
		Arrays.sort(down_right);
		Arrays.sort(down);
		
		for(int i = 0 ; i < (size*(size)) ; i++){
			
			if (isMember (column, up)) {
				if (flag.isAvailable (row-1, column-1)) {
					codewordsStream[flagStream] = symbolReceived[row-1][column-1];
					flagStream ++;
				}
				if (isMember(column, up_right)) {
					column -- ;
				} else {
					column ++ ;
					row -- ;
					// Out of upper-bound detection
					if (row == 0) {
						row = 1 ;
						// Skip the timing column.
						if (column == 9) {
							column = column - 3;
						} else {
							column = column - 2;
						}
					}
				}
			}else if (isMember (column, down)) {
				if (flag.isAvailable (row-1, column-1)) {
					codewordsStream[flagStream] = symbolReceived[row-1][column-1];
					flagStream ++;
				}
				if (isMember (column, down_right)) {
					column --;
				} else {
					column ++;
					row ++;
					// Out of lower-bound detection.
					if (row == (size + 1)) {
						row = size;
						column = column - 2;
					}
				}
			}
		}
	}
	
	/*
	 * Decodes the codewords stream with the RS decoder,
	 * restores the mode and data length,
	 * returns the input message.
	 */
	public String decodeData () {

		int [] codewordsRSInt = booleanToIntArray(codewordsStream);
		
		// RS decoding
		int [] syndrome = rs.rsCalcSyndrome(codewordsRSInt, nbSymb);
		int [] position = rs.rsFindErrors(syndrome, codewordsRSInt.length);		
		int [] codewordsData;
		
		if (position != null) { // There are errors to correct
			if (position.length == 0) {
				nbErrors = 0;
			}
			else {
				nbErrors = position.length;
			}
			codewordsData = rs.rsCorrectErrata(codewordsRSInt, syndrome, position);	
		}
		
		else  { // No errors to correct or cannot find positions of errors
			nbErrors = -1;

			codewordsData = new int[codewordsRSInt.length];
			for(int i = 0 ; i < codewordsRSInt.length ; i++) {
				codewordsData[i] = codewordsRSInt[i];
			}
		}
		
		int[] codewordsDataPadded = new int[dataCodewords];
		for(int i = 0 ; i < dataCodewords ; i++) {
			codewordsDataPadded[i] = codewordsData[i];
		}
		
		boolean[] cwDataBoolean = intArrayToBoolean(codewordsData);	
		
		// Read the mode
		boolean[] modeBoolean = new boolean [4];
		for (int i = 0 ; i < 4 ; i++) {
			modeBoolean [i] = cwDataBoolean[i];
		}
		for (int i = 0 ; i < 4 ; i++) {
			boolean temp = modeBoolean[i];
			if (temp == true) { // True is 1.
				mode += Math.pow(2, 3-i); 
			}
		}
		
		
		// Read the data length
		boolean[] countBoolean = new boolean [8];
		for(int i = 4 ; i < 12 ; i++) {
			countBoolean[i-4] = cwDataBoolean[i];

		}
		
		for (int i = 0 ; i < 8 ; i++) {
			boolean temp = countBoolean[i];
			if (temp == true) { // True is 1.
				count += Math.pow(2, 7-i); 		
			}
		}
		String msgOut;
		
		if(count > MAX_DATA) {
			Log.e("QRDecoder","Payload size exceeded.");
			msgOut = null;
		}
		else {
			boolean[] dataTemp = new boolean [count*8];
			for(int i = 12 ; i < count*8+12 ; i++) {
				dataTemp [i-12] = cwDataBoolean [i];
			}
			int [] dataInt = booleanToIntArray(dataTemp);
			this.dataOut = new byte [dataInt.length];
			for(int i = 0 ; i < dataInt.length ; i++) {
				this.dataOut [i] = (byte) dataInt [i];
			}

			StringBuilder outputData = new StringBuilder();
			for(int i = 0 ; i < dataInt.length ; i++) {
				outputData.append ((char) dataInt[i]);
			}
			msgOut = outputData.toString();
		}
		
		return msgOut;
	}
	
	public byte[] returnValue () {
		return this.dataOut;
	}
	
	public int getNumberErrors () {
		return this.nbErrors;
	}
	
	// Transform an integer array into a boolean array.
	public boolean[] intArrayToBoolean (int[] msg) {
		boolean[] msgStreamBoolean = new boolean[msg.length*8];
		for(int i = 0 ; i < msg.length ; i ++) {
			int value = msg[i];
			if (value < 0 ) {
				value &= 255;
			}
			for(int j = i*8 ; j < (i+1)*8 ; j++) {
				if(((byte)value & 128) == 0) {
					msgStreamBoolean[j] = false; // 0 is false
				}
				else {
					msgStreamBoolean[j] = true;
				}
			    value <<= 1;
			}
		}
		return msgStreamBoolean;
	}
		
	// Divide the message into 8-bit codewords.
	public int[] booleanToIntArray(boolean[] msg) {
		int length = 0;
		if(msg.length % 8 == 0) {
			length = msg.length/8;
		} else {
			length = (int)Math.floor(msg.length/8) +1;
		}
		int[] msgStreamInt = new int[length];
		for (int i = 0 ; i < length ; i++) {
			boolean [] temp = new boolean [8];
			for(int j = 8*i ; j < 8*(i+1) ; j++) {
				temp [j - 8*i] = msg[j];
			}
			if (temp [0] == false) { // positive number
				for (int pos1 = 1 ; pos1 < 8 ; pos1 ++) {
					if (temp[pos1] == true) { // True is 1.
						msgStreamInt[i] += Math.pow(2, 7-pos1); 
					}
				}
			}
			else {
				for (int pos1 = 1 ; pos1 < 8 ; pos1 ++) { // negative number
					if (temp[pos1] == true) { // True is 1.
						msgStreamInt[i] += Math.pow(2, 7-pos1); 
					}
				}
				msgStreamInt[i] = - (128 - msgStreamInt[i]);
			}
		}
		return msgStreamInt;
	}
	
	private boolean isMember (int value, int[] array) {
		/* Checks if the Value is member of the array
		 * Output: boolean: true if value exists in array
		 * @Thomas
		 */
		return (Arrays.binarySearch(array, value) > 0);
	}
	
	private boolean xor(boolean a, boolean b) {
		/*XOR operation
		 * @Thomas
		 */
		return (a && !b) || (b && !a);
	}
		
	// Convert the boolean array (15 long) into a 2-byte array. Add a zero for the left-most bit -> 16 long.
	public byte[] booleanToByteArray(boolean[] msgBoolean) { // TEST OK
		byte[] msgByte = new byte[2];
		int pos1 = 0;
		int pos2 = 0;
		for (int i = 0 ; i < 7 ; i++) {
			boolean temp = msgBoolean[i];
			if (temp == true) { // True is 1.
				msgByte[0] += Math.pow(2, 6-pos1); 
			}
			pos1++;
		}
		for (int i = 7 ; i < 15 ; i++) {
			boolean temp = msgBoolean[i];
			if (temp == true) { // True is 1.
				msgByte[1] += Math.pow(2, 7-pos2); 
			}
			pos2++;
		}
		return msgByte;
	}
	
	// Convert a 2-byte array into a positive integer
	public int byteArrayToInt (byte[] byteArray) { // TEST OK
		int value = 0;
		value = (int) ((byteArray[0] << 8)	+ byteArray[1]);	
		if (value < 0) {
			value += 65792;
		}
		return value;
	}
	
	
}
