package se.kth.android.projectred.QR;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Queue;



import android.util.Log;

public class QRencoderV2 {
	
	
	//CONSTANTS
	byte[] pad_basis = {(byte) 0xEC, 0x11};
	byte BYTE_MODE=0x04;
	byte TERMINATOR =0x00;
	int SIZEVERSION = 25;
	
	//ASSUMING V2 
	int data_num_cw =16; //error correction level H
	byte data_bits= 0x11;//error correction level H
	byte MAX_PAYLOAD = 14;//error correction level H
	int size = SIZEVERSION;
	int numSymb = 28; // The number of correction bytes for error correction level H
	boolean[] format = new boolean[5];
	int formatInt; // 10001 for high level and mask number 1
	
	//VARIABLES
	//QR code data 8 bits per byte
	byte[] data;
	boolean[] msg_stream;
	boolean[] msgStreamEncoded;
	byte character_count_indicator;
	boolean QR_Mat[][];
	GenMask mask;
	GenFlag flag;
	ReedSolomon rs;
	InitMatrix matInit;
		
	public QRencoderV2(byte[] d){
	/*Constructor of the QRencoder class. Takes the data for the QR code as an input
	 *@Thomas
	 */
		// Reed Solomon encoder and decoder
		rs = new ReedSolomon();
		matInit = new InitMatrix(2);
		QR_Mat = matInit.getMatrix();
		data = d;
		character_count_indicator=(byte) data.length;
		if (character_count_indicator < MAX_PAYLOAD)
			make_msg_stream();
		else
			Log.e("QREncoder","Payload size exceeded");
		
		//Set flag and mask to version 2
		mask=new GenMask(2);
		flag=new GenFlag(2);
	}

	
	boolean bit(byte data, int pos){
		/*Returns one bit from data
		 * Input:
		 * 		data: one byte
		 * 		pos: position you want to know the bit
		 * Output:
		 * 		boolean: data(pos)
		 * @Thomas
		 */
		byte displayMask = (byte) (1 << 7);
		data <<= pos;
	return ( data & displayMask ) != 0;      	
	}
	 
	public void make_msg_stream(){
		/*This function creates the message stream from the data
		 * msg_stream= BYTE_MODE - character_count - data - terminator - padding
		 * @Thomas
		 */
		msg_stream=new boolean[data_num_cw*8];
		
		//Write ByteMode
		msg_stream[0]=bit(BYTE_MODE,4); //First bit is set
		msg_stream[1]=bit(BYTE_MODE,5);
		msg_stream[2]=bit(BYTE_MODE,6);
		msg_stream[3]=bit(BYTE_MODE,7);
		//Write CharCount
		msg_stream[4]=bit(character_count_indicator,0);
		msg_stream[5]=bit(character_count_indicator,1);
		msg_stream[6]=bit(character_count_indicator,2);
		msg_stream[7]=bit(character_count_indicator,3);
		
		msg_stream[8]=bit(character_count_indicator,4);
		msg_stream[9]=bit(character_count_indicator,5);
		msg_stream[10]=bit(character_count_indicator,6);
		msg_stream[11]=bit(character_count_indicator,7);
		//Write Data
		for(int i=12;i<character_count_indicator*8+8+4;i++){ //start at 8+4
			msg_stream[i]=bit(data[(int) (Math.floor((i-4)/8)-1)],(i-4)%8);
		}
		// Add terminator
		for(int i=character_count_indicator*8+8+4;i<character_count_indicator*8+8+4+4;i++){  //start at 8+4+cw*8
			msg_stream[i]=bit(character_count_indicator,4+i);
		}
		
		//Adding Padding at the end
		int nextBit=character_count_indicator*8+8+4+4;
		int required_pad_cw=data_num_cw-(character_count_indicator+2); // ByteMode, CharCount and Terminator add 2 CWs
		for (int i=0;i<required_pad_cw;i++){
			for (int j=0;j<8;j++){
				byte padding=pad_basis[i%2];
 				msg_stream[nextBit++]=bit(pad_basis[i%2],j);
			}
		}
		
		// Chunck the boolean array in integer codewords.
		int[] msgStream = booleanToIntArray(msg_stream);

		// Encode the data
		int[] msgOut = rs.rsEncodeMsg(msgStream, numSymb);

		for(int i = 0 ; i < msgOut.length ; i++) {
			if(msgOut[i] < 0) {
				msgOut[i] &= 255;
			}
		}
		
		// Transform the message encoded to a boolean array before filling in the matrix.
		msgStreamEncoded = intArrayToBoolean(msgOut);
				
	}
	
	public void addFormatInfo(){
		/*Requires bchenc() so for now it is just the output calculated in Matlab
		 * @Thomas
		 */
		// Create the format for v2, level correction H, mask 1. (10 001)
		
		int formatUnmasked = rs.qrEncodeFormat(formatInt); // this format unmasked is the same as matlab -> pb NOT in encoding!
		
		byte[] formatUnmaskedByte = intToByteArray(formatUnmasked);
		byte[] maskFormat = {(byte)0x54, (byte)0x12};
		byte[] formatMaskedByte = new byte[2];
		formatMaskedByte[0] = (byte) ((byte)formatUnmaskedByte[2] ^ (byte) maskFormat[0]);
		formatMaskedByte[1] = (byte) ((byte)formatUnmaskedByte[3] ^ (byte) maskFormat[1]);		

		formatMaskedByte[0] ^= 255;
		formatMaskedByte[1] ^= 255;
		
		// To boolean array
		boolean[] format_masked = new boolean[15];
		for(int i = 0 ; i < 7 ; i++) {
			format_masked[i] = bit(formatMaskedByte[0], i+1);
		}
		
		for(int i = 0 ; i < 8 ; i++) {
			format_masked[i+7] = bit(formatMaskedByte[1], i);
		}
		
		// Invert the mask
		boolean[] format_masked_inv = new boolean [15];
		for(int i = 0; i < 15; i++){
			format_masked_inv[i] = format_masked[14 - i];
		}	
		
		for (int row=0 ; row < 6 ; row++){
			int col = 8;
			QR_Mat[row][col] = format_masked_inv[row];
		}
		for (int row=7;row<9;row++){
			int col=8;
			QR_Mat[row][col] = format_masked_inv[row-1];
		}
		for (int row=18;row<25;row++){
			int col=8;
			QR_Mat[row][col]= format_masked_inv[row - 10];
		}
		
		for (int col=0;col<6;col++){
			int row=8;
			QR_Mat[row][col] = format_masked[col];
		}
		
			QR_Mat[8][7] = format_masked[6];
		
		for (int col=17;col<25;col++){
			int row=8;
			QR_Mat[row][col] = format_masked[col - 10];
		}
		
	}
	
	public void dataPlacement(){
		/*Places the bitstream inside the QR_Mat according to Version2
		 * msg_stream[] -> QR_Mat[][]
		 * @Thomas
		 */
		
		//CONSTANTS
		int up_left[] = {24, 20, 16, 12, 8, 3};
		int up_right[] = {25, 21, 17, 13, 9, 4};
		int up[] = {24, 20, 16, 12, 8, 3, 25, 21, 17, 13, 9, 4};
		int down_left[] = {22, 18, 14, 10, 5, 1};
		int down_right[] = {23, 19, 15, 11, 6, 2};
		int down[] = {22, 18, 14, 10, 5, 1, 23, 19, 15, 11, 6, 2};
	
		//VARIABLES
		int row=size;
		int column=size;
		int indx=0;
		
		//Sort Arrays to allow for binary search
		//Search command
		//if (Arrays.binarySearch(up, 1) > 0)
		Arrays.sort(up_left);
		Arrays.sort(up_right);
		Arrays.sort(up);
		Arrays.sort(down_left);
		Arrays.sort(down_right);
		Arrays.sort(down);
		
		
		for(int k=0;k<(size*(size));k++){
			
				if (isMember(column,up)){
					if (flag.isAvailable(row-1, column-1))
						QR_Mat[row-1][column-1]=msgStreamEncoded[indx++];
					if (isMember(column,up_right)){
						column--;
					}else{
						column++;
						row--;
						// Out of upper-bound detection
						if(row==0){
							row=1;
							// Skip the timing column.
							if (column==9){
								column=column-3;
							}else{
								column=column-2;
							}
						}
					}
				}else if (isMember(column,down)){
					if (flag.isAvailable(row-1, column-1))
						QR_Mat[row-1][column-1]=msgStreamEncoded[indx++];
					if (isMember(column,down_right)){
						column--;
					}else{
						column++;
						row++;
						// Out of lower-bound detection.
						if (row==(size+1)){
							row=size;
							column=column-2;
						}
					}
				}
			}
	}
	
	public void applyMask(int mask_Nr){
		/*Applies the selected Mask to the QR_Mat[][]
		 * @Thomas
		 */
		adaptFormatNb(mask_Nr);
		for (int row=0;row<size;row++){
			for (int col=0;col<size;col++){
				if (flag.isAvailable(row, col)) {
					QR_Mat[row][col]=xor(QR_Mat[row][col],mask.getElement(row, col, mask_Nr));
				}
			}
		}
	}
		
	public void adaptFormatNb (int maskNb) {
		switch (maskNb) {
		case 0:
			this.formatInt = 16;
			break;
		case 1:
			this.formatInt = 17;
			break;
		case 2:
			this.formatInt = 18;
			break;
		case 3:
			this.formatInt = 19;
			break;
		case 4:
			this.formatInt = 20;
			break;
		case 5:
			this.formatInt = 21;
			break;
		case 6:
			this.formatInt = 22;
			break;
		case 7:
			this.formatInt = 23;
			break;
		default: // apply mask 1 by default (as in Matlab)
			this.formatInt = 17;
			break;
		}	
	}
	
	private boolean xor(boolean a, boolean b) {
		/*XOR operation
		 * @Thomas
		 */
		return (a && !b) || (b && !a);
	}


	public boolean[][] getQR_mat(){
		return QR_Mat;
	}
	
	private boolean isMember(int value, int[] array){
		/* Checks if the Value is member of the array
		 * Output:
		 * 		boolean: true if value exists in array
		 */
		return (Arrays.binarySearch(array, value) > 0);
	}
	
	boolean[] getMessage(){
		return msg_stream;
	}
	
	// Divide the message into 8-bit codewords.
	public int[] booleanToIntArray(boolean[] msg) {
		int[] msgStreamInt = new int[data_num_cw];
		boolean [] temp = new boolean [8];

		for (int i = 0 ; i < data_num_cw ; i++) {
			for(int j = 8*i ; j < 8*(i+1) ; j++) {
				temp [j-i*8] = msg[j];
			}
			
			if (temp [0] == false) { // positive number
				for (int pos = 1 ; pos < 8 ; pos++) {
					if (temp [pos] == true) { // True is 1.
						msgStreamInt[i] += Math.pow(2, 7-pos);
					}
				}
			}
			else { // negative number
				for (int pos = 1 ; pos < 8 ; pos++) {
					if (temp [pos] == true) { // True is 1.
						msgStreamInt[i] += Math.pow(2, 7-pos);
					}
				}
				msgStreamInt [i] = - (128 - msgStreamInt [i]);
			}
		}
		return msgStreamInt;
	}
	
	// Transform an integer array into a boolean array.
	public boolean[] intArrayToBoolean (int[] msg) {
		boolean[] msgStreamBoolean = new boolean[msg.length*8];
		for(int i = 0 ; i < msg.length ; i ++) {
			int value = msg[i];
			if (value < 0) {
				value &= 255;
			}
			for(int j = i*8 ; j < (i+1)*8 ; j++) {
				if(((byte)value & 128) == 0) {
					msgStreamBoolean[j] = false; // 0 is false
				}
				else {
					msgStreamBoolean[j] = true;
				}
			    value <<= 1;
			}
		}
		return msgStreamBoolean;
	}
	
	// Transform an integer to a byte array
	public static byte[] intToByteArray(int value) {
        byte[] b = new byte[4];
        for (int i = 0; i < 4; i++) {
            int offset = (b.length - 1 - i) * 8;
            b[i] = (byte) ((value >>> offset) & 0xFF);
        }
        return b;
    }


}