package se.kth.android.projectred.QR;

import java.util.ArrayList;

import android.util.Log;

/**
 * This class defines the operations in a Galois field: logarithmic multiplication, division, polynomial operations.
 * Created Tuesday 16th, 2013.
 * Modified Wednesday 17th, 2013. Thursday 18th, 2013.
 * @author Claire
 *
 */
public class ReedSolomon {
	//CONSTANT 
	static final boolean DEBUG=false;
	// Class variables
	int[] gfExp = new int[512];
	int[] gfLog = new int[256];
	
	// Constructor
	public ReedSolomon() {
		this.alphaChart();
	}
	
	/*****************************************************************************************************************************************************************
	 * DEFINITIONS OF OPERATIONS IN THE GALOIS FIELD. 
	 * ESSENTIAL FOR ALL THE FOLLOWING DATA PROCESSING.
	 *****************************************************************************************************************************************************************/
	
	/**
	 * Constructs table of logarithms, contains the power of alpha.
	 */
	public void alphaChart() {	//  TEST OK!
		for(int i = 0 ; i < 255 ; i++) {
			gfLog[i] = 0;
		}
		
		int x = 1;
		gfExp[0] = 1;
		
		for (int i = 1 ; i < 255 ; i++) {
			x =  (int) ((byte) x << 1); // left shift one bit, unsigned bytes
			//System.out.println("x " + x);
			if(x < 0) {
				x = (int) ((byte) x ^ (byte) 0x11d); // XOR operations with modulus 100011101
				if (x < 0) {
					x = x & 255;
				}
				this.gfExp[i] = x;
				this.gfLog[x] = i;
			}
			else {
				this.gfExp[i] = x;
				this.gfLog[x] = i;
			}
		}
		for(int i = 255 ; i < 512 ; i++) {
			this.gfExp[i] = this.gfExp[i-255];
		}
	}
	
	/**
	 * Definition of multiplication with logarithms.
	 * @param x
	 * @param y
	 * @return x*y
	 */
	public int gfMultiplication (int x , int y) { //  TEST OK!
		if (x == 0 || y == 0) {
			return 0;
		}
		if (x < 0 && y > 0) {
			x &= 255;
			return this.gfExp[this.gfLog[x] + this.gfLog[y]];
		}
		if (y < 0 && x > 0) {
			y &= 255;
			return this.gfExp[this.gfLog[x] + this.gfLog[y]];
		}
		if (x < 0 && y < 0) {
			x &= 255;
			y &= 255;
			return this.gfExp[this.gfLog[x] + this.gfLog[y]];
		}
		else {
			return this.gfExp[this.gfLog[x] + this.gfLog[y]];
		}
	}
	
	/**
	 * Definition of division.
	 * @param x
	 * @param y
	 * @return x/y
	 */
	public int gfDivision (int x , int y) {
		if (y == 0) {
			System.out.println("Arithmetic exception, division by zero.");
			return -1;
		}
		if (x == 0) {
			return 0;
		}
		else {
			if(x < 0) {
				x &= 255;
			}
			if(y < 0) {
				y &= 255;
			}
			int temp = gfLog[x] - gfLog[y];
			if(temp < 0)
				temp += 255;
			return gfExp[temp]; // positive difference
		}
	}
	
	/** Definition of the Galois field polynomial operations.
	 * A polynomial is defined as a list of numbers, in descending power of orders x.
	 * 2*x^4 + x^2 <=> [2, 0, 1, 0, 0]  
	 */
	
	/**
	 * Multiplication of a polynomial by a scalar.
	 * @param polynomial
	 * @param scalar
	 * @return r: polynomial multiplied by scalar
	 */
	public int[] gfPolyScalar(int[] polynom, int scalar) { // TEST ok!
		int[] p = new int[polynom.length];
		for(int i = 0 ; i < polynom.length ; i++) {
			p[i] = gfMultiplication(polynom[i], scalar);
		}
		return p;
	}

	/**
	 * "Addition" of two polynomials. Using XOR.
	 * @param p: first polynomial
	 * @param q, second polynomial
	 * @return p+q (XOR sense)
	 */
	public int[] gfPolyAdd(int[] p, int[] q) { // Test ok!
		int[] r = new int[Math.max(p.length, q.length)];
		for(int i = 0 ; i < p.length ; i++) {
			r[i + r.length - p.length] = p[i];
		}
		for(int i = 0 ; i < q.length ; i++) {
			r[i + r.length - q.length] = (int) ((byte) r[i + r.length - q.length] ^ (byte) q[i]);
		}
		return r;
	}
	
	/**
	 * Multiplication of two polynomials.
	 * @param p polynomial
	 * @param q polynomial
	 * @return p*q polynomial
	 */
	public int[] gfPolyMultiply(int[] p , int[] q) { // Test ok!
		int[] r = new int[p.length + q.length -1];
		for(int i = 0 ; i < p.length ; i++) {
			for(int j = 0 ; j < q.length ; j++) {
				r[i+j] = (int) ((byte) r[i+j] ^ (byte) gfMultiplication(p[i], q[j])) & 255; // XOR "addition"
			}
		}
		return r;
	}
	
	/**
	 * Evaluate polynomial in a particular value of x.
	 * @param p, polynomial
	 * @param value at which evaluate the polynomial
	 * @return polynomial p evaluated in value -> integer y
	 */
	public int gfPolyEval(int[] p, int value) {
		int y = p[0];
		for(int i = 1 ; i < p.length ; i++) {
			y = (int)((byte)gfMultiplication(y, value) ^ (byte)p[i]);
		}
		return y;
	}
	
	/*****************************************************************************************************************************************************************
	 * REED SOLOMON ENCODER. 
	 *****************************************************************************************************************************************************************/
	
	/**
	 * Reed-Solomon generator polynomial.
	 * @param numSymb, the number of error correction symbols
	 * @return the generator polynomial
	 */
	public int[] rsPolyGenerator(int numSymb) { // TEST ok
		ArrayList<Integer> generatorArray = new ArrayList<Integer>();
		generatorArray.add(0, 1);
		
		int[] array = new int[2];
		array[0] = 1;
		
		for(int i = 0 ; i < numSymb ; i++) {
			array[1] = gfExp[i];
			/*for(int j = 0 ; j < generatorArray.size() ; j++) {
				System.out.println("j " + j + " generator " + generatorArray.get(j));
			}**/
			int[] result = gfPolyMultiply(arrayToInt(generatorArray), array);
			generatorArray = intToArray(result, generatorArray);
			/*for(int j = 0 ; j < result.length ; j++) {
				System.out.println("j " + j + " result " + result[j]);
				System.out.println("j " + j + " generator array " + generatorArray.get(j));

			}**/
		}
		return arrayToInt(generatorArray);
	}
	
	/**
	 * Transforms an arrayList to an integer array.
	 * @param array to transform
	 * @return integer array
	 */
	public int[] arrayToInt(ArrayList<Integer> array) {
		int[] r = new int[array.size()];
		for(int i = 0 ; i < array.size() ; i++) {
			r[i] = array.get(i);
		}
		return r;
	}
	
	/**
	 * Transforms an integer array to an arrayList.
	 * @param r, integer array to transform
	 * @param array, the arrayList to fill in with r values.
	 * @return array.
	 */
	public ArrayList<Integer> intToArray(int[] r, ArrayList<Integer> array) {
		array.clear();
		for(int i = 0 ; i < r.length ; i++){
			array.add(i, r[i]);		
		}
		return array;
	}
	
	/**
	 * Encoding a message with the Reed-Solomon method.
	 * @param msgIn, the message to encode.
	 * @param numSymb, the error correction.
	 * @return the message encoded.
	 */
	public int[] rsEncodeMsg(int[] msgIn, int numSymb) { // TEST ok
		int[] generator = rsPolyGenerator(numSymb);
		int[] msgOut = new int[msgIn.length + numSymb];
		for(int i = 0 ; i < msgOut.length ; i++) {
			msgOut[i] = 0;
		}
		for(int i = 0 ; i < msgIn.length ; i++) {
			msgOut[i] = msgIn[i];
		}
		for(int  i = 0 ; i < msgIn.length ; i++) {
			int coeff = msgOut[i];
			if(coeff != 0) {
				for(int j = 0 ; j < generator.length ; j++) {
					msgOut[i+j] = (int) ((byte) msgOut[i+j] ^ (byte)gfMultiplication(generator[j], coeff));
				}
			}
		}
		for(int i = 0 ; i < msgIn.length ; i++) {
			msgOut[i] = msgIn[i];
		}
		/*
		for(int i = 0 ; i < msgOut.length ; i++) {
			System.out.println("msg out " + msgOut[i]);
		}**/
		return msgOut;
	}
	
	/*****************************************************************************************************************************************************************
	 * REED SOLOMON DECODER, ERROR DETECTION AND CORRECTION.
	 *****************************************************************************************************************************************************************/
	
	/**
	 * To decode, first need to calculate the syndrome of the message.
	 * If syndrome == 0 -> msg is undamaged.
	 * Else, damage and need to correct the msg.
	 * @param msg
	 * @param nbSymb
	 * @return the syndrome
	 */
	public int[] rsCalcSyndrome(int[] msg, int nbSymb) { // TEST ok!
		int[] syndrome = new int[nbSymb];
		for(int i = 0 ; i < syndrome.length ; i++) {
			syndrome[i] = 0;
		}
		for(int i = 0 ; i < nbSymb ; i++) {
			syndrome[i] = gfPolyEval(msg, gfExp[i]);
			if(syndrome[i] < 0) {
				syndrome[i] &= 255;
			}
			//System.out.println("syndrome " + syndrome[i]);
		}
		return syndrome;
	}
	
	/**
	 * Erasure correction. Correct the errors when their positions are known.
	 * @param msg
	 * @param syndrome
	 * @param position of the errors (supposed to be known)
	 */
	 public int[] rsCorrectErrata(int [] msg, int[] syndrome, int[] position) { // TEST OK!
		 // Calculate error locator polynomial
		 ArrayList<Integer> q = new ArrayList<Integer>();
		 q.add(0, 1);
		 int[] temp = new int[2];
		 temp[1] = 1;
		 int[] p = new int[position.length];
		 ArrayList<Integer> qprime = new ArrayList<Integer>();
		 int[] r = new int[position.length];

		 for(int i = 0 ; i < position.length ; i++) {
			 int x = gfExp[msg.length-1-position[i]];
			 temp[0] = x;
			 int[] qtemp = gfPolyMultiply(arrayToInt(q), temp);
			 q = intToArray(qtemp, q);
		 }
		 // Calculate error correction polynomial
		 for(int i = 0 ; i < position.length ; i++) {
			 p[position.length - 1 - i] = syndrome[i]; // reverse
		 }
		 int[] pq = gfPolyMultiply(p, arrayToInt(q));
		 for(int i = pq.length-position.length ; i < pq.length ; i++) {
			 r[i - pq.length + position.length] = pq[i];
		 }
		 // Formal derivative of error locator eliminates even terms
		 for(int i = (q.size() & 1) ; i < q.size() ; i+=2) {
			 qprime.add(q.get(i));
		 }

		 // Compute corrections
		 for(int i = 0 ; i < position.length ; i++) {
			 int value1 = gfExp[(position[i] - msg.length) & 255];
			 int value2 = gfPolyEval(r, value1);
			 int value3 = gfPolyEval(arrayToInt(qprime), gfMultiplication(value1, value1));
			 msg[position[i]] = (int) ((byte) msg[position[i]] ^ (byte) gfDivision(value2, gfMultiplication(value1, value3)));
		 }
		 return msg;
	 }
	 
	 /**
	  * Find the error positions when they are unknown.
	  * @param syndrome
	  * @param lengthMsg, the message's length.
	  * @return integer array containing the error positions.
	  */
	 public int[] rsFindErrors(int[] syndrome, int lengthMsg) { // TEST ok
		 // Find error locator polynomial with Berlekamp-Massey algorithm
		 ArrayList<Integer> errorPoly = new ArrayList<Integer>();
		 errorPoly.add(1);
		 ArrayList<Integer> oldPoly = new ArrayList<Integer>();
		 oldPoly.add(1);
		 ArrayList<Integer> errorPositions = new ArrayList<Integer>();
		 for (int i = 0 ; i < syndrome.length ; i++) {
			 oldPoly.add(0);
			 int delta = syndrome[i];
			 for (int j = 1 ; j < errorPoly.size() ; j++) {
				 delta = (int) ((byte) delta ^ (byte) gfMultiplication(errorPoly.get(errorPoly.size() - 1 - j), syndrome[i-j]));
			 }
			 if (delta != 0) {
				 if (oldPoly.size() > errorPoly.size()) {
					 int [] newPoly = gfPolyScalar(arrayToInt(oldPoly), delta);
					 oldPoly = intToArray(gfPolyScalar(arrayToInt(errorPoly), gfDivision(1, delta)), oldPoly);
					 errorPoly = intToArray(newPoly, errorPoly);
				 }
				 errorPoly = intToArray(gfPolyAdd(arrayToInt(errorPoly), gfPolyScalar(arrayToInt(oldPoly), delta)), errorPoly);
			 }
		 }
		 int errs = errorPoly.size() - 1;
		 if (errs*2 > syndrome.length) {
			 Log.e("Error RS", "Too many errors to correct.");
			 return null; // to many errors to correct
		 }
		 // Find zeros of error polynomial
		 for (int i = 0 ; i < lengthMsg ; i++) {
			 if (gfPolyEval(arrayToInt(errorPoly), gfExp[255-i]) == 0) {
				 errorPositions.add(lengthMsg - 1 - i);
			 }
		 }
		if(DEBUG)Log.d("RS", "RS error positions " + errorPositions.size());
		if(DEBUG)Log.d("RS", "RS errs " + errs);

		 if (errorPositions.size() != errs) {
			 if(DEBUG)Log.e("Error RS", "Could not find error locations.");
			 return null;
		 }	 
		 return arrayToInt(errorPositions);
	 }
	 
	 /*****************************************************************************************************************************************************************
	  * ENCODING AND DECODING OF THE FORMAT BITS. 
	  *****************************************************************************************************************************************************************/
	 /**
	  * Divide the format by the generator of the code. The generator is always the same for a 5-bit format: (byte) 10100110111 = (integer) 1335 = 0x537.
	  * The remainder is zero if the format is undamaged.
	  * @param format
	  * @return the format
	  */
	 public int qrCheckFormat(int format) { // test OK!
		 int generator = (int) 0x537;
		 for (int i = 4 ; i >= 0 ; i--) {
			 if ((Math.pow(2, i+10) <= format) && (format <  Math.pow(2, i+11))) {
				 format = format ^ (generator << i);				 
			 }
		 }
		 return format;
	 }
	 
	 /**
	  * Function that encodes the format. 
	  * @param format: 5-bit format (2-bit format level (L, M, Q, H) and 3-bit mask type).
	  * @return format encoded: 5 left bits are the initial format and 10 right bits are the redundancy for correction.
	  */
	 public int qrEncodeFormat(int format) { // test OK!
		 return ((format << 10) ^ qrCheckFormat(format << 10));
	 }
	 
	 /**
	  * Calculating the Hamming distance for x.
	  * @param x: the result of the XOR operation between the bits sequences to be compared.
	  * @return the number of bits that are different = Hamming weight.
	  */
	 public int hammingWeight (int x) {
		 int weight = 0;
		 while (x > 0) {
			 if (x % 2 != 0)
				 weight ++; // counting the number of 1 (1 means a difference in the bits when doing XOR)
			 x = x >> 1;
		 }
		 return weight;
	 }
	 
	 /**
	  * Function that decodes the format. Since there are only 32 possible format codes,
	  * the Hamming distance is computed between the possible formats and the format to decode. The format with the smallest distance is retained.
	  * @param format
	  * @return the format decoded
	  */
	 public int qrDecodeFormat(int format) { // Test OK!
		 int bestFormat = -1;
		 int bestDistance = 15;
		 for (int testFormat = 0 ; testFormat < 32 ; testFormat ++) { // computing the Hamming distance between the encoded format and the 32 possible encoded formats. (5-bit)
			 int testCode = qrEncodeFormat(testFormat);
			 int testDistance = hammingWeight(format ^ testCode);
			 if (testDistance < bestDistance) {
				 bestDistance = testDistance;
				 bestFormat = testFormat;
			 }
			 
			 else {
				 if (testDistance == bestDistance) { // the format is close to two possible formats with the same hamming distance.
					 bestFormat = -1;
				 }
			 }
			 //System.out.println("Boucle " + testFormat + " bestDist " + bestDistance + " bestFormat " + bestFormat);
		 }
		 if (bestFormat == -1) {
			 Log.e("RS", " RS: Too many errors, cannot identify the format unambiguously.");
		 }
		 return bestFormat;
	 }
	 
	
}


