package se.kth.ict.iv1201.validation.view;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import javax.validation.Constraint;
import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import javax.validation.Payload;

/**
 * The annotated target is checked to be a valid zip code.
 */
@Constraint(validatedBy = ValidZipCode.ZipCodeValidator.class)
@Documented
@Target({ElementType.METHOD, ElementType.FIELD, ElementType.ANNOTATION_TYPE})
@Retention(RetentionPolicy.RUNTIME)
public @interface ValidZipCode {

    String message() default "Not a valid zip code.";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    class ZipCodeValidator implements ConstraintValidator<ValidZipCode, String> {

        @Override
        public void initialize(ValidZipCode constraintAnnotation) {
        }

        @Override
        public boolean isValid(String value, ConstraintValidatorContext context) {
            if (isEmpty(value, context)) {
                return false;
            }
            try {
                Integer.parseInt(value);
            } catch (NumberFormatException nfe) {
                return false;
            }
            return value.length() == 5;
        }

        private boolean isEmpty(String value, ConstraintValidatorContext context) {
            if (value.length() == 0) {
                context.disableDefaultConstraintViolation();
                context.buildConstraintViolationWithTemplate("Zip code must be entered").addConstraintViolation();
                return true;
            }
            return false;
        }
    }
}
